<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\ResourceModel\Item\CollectionFactory;
use Magento\Store\Model\StoreManagerInterface;

class CatalogImport extends \Ecomero\ErpCore\Model\Executor
{
    protected $settings;
    protected $storeManager;
    protected $itemCollectionFactory;
    protected $erp;
    protected $simpleProduct;
    protected $configurableProduct;
    protected $isImportRunning = false;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpCatalogInterface $erp,
        Settings $settings,
        CronLock $cronLock,
        StoreManagerInterface $storeManager,
        CollectionFactory $itemCollectionFactory,
        SimpleProduct $simpleProduct,
        ConfigurableProduct $configurableProduct
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->erp = $erp;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->itemCollectionFactory = $itemCollectionFactory;
        $this->simpleProduct = $simpleProduct;
        $this->configurableProduct = $configurableProduct;

        $this->erp->setLogger($logger);
    }

    public function isImportRunning()
    {
        return $this->isImportRunning;
    }

    protected function run(bool $force): string
    {
        return $this->import($force);
    }

    protected function getServiceDescription(): string
    {
        return 'product import';
    }

    protected function getCapability(): string
    {
        return Capability::PRODUCT_IMPORT;
    }

    private function getUniqueProducts(object $itemCollection): array
    {
        $uniqueProdArray = [];

        foreach ($itemCollection->getItems() as $item) {
            if ('' == $item->getCommmonName()) {
                continue;
            }
            if (false == array_key_exists($item->getCommmonName(), $uniqueProdArray)) {
                $uniqueProdArray[$item->getCommmonName()] = $item;
            }
            if (null != $item->magentoId) {
                $uniqueProdArray[$item->getCommmonName()]->associatedProductIds[] = $item->magentoId;
            }
        }

        return $uniqueProdArray;
    }

    private function createProducts(\Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection): void
    {
        if ($itemCollection->count() > 0) {
            $this->simpleProduct->createOrUpdateSimpleProducts($itemCollection);

            $uniqueProducts = $this->getUniqueProducts($itemCollection);
            foreach ($uniqueProducts as $item) {
                $this->configurableProduct->createOrUpdateConfigurableProduct($item);
            }

            if ($this->settings->isDisableRemovedProducts()) {
                $this->simpleProduct->disableRemovedProducts($itemCollection);
            }
        }
    }

    private function import(bool $force): string
    {
        $this->isImportRunning = true;

        $websiteName = '';
        $errorMessage = '';

        try {
            $this->storeManager->setCurrentStore(0);
            $websiteName = null;
            $allWebsites = $this->storeManager->getWebsites(false, true);
            $itemCollection = $this->itemCollectionFactory->create();
            foreach ($allWebsites as $website) {
                $websiteId = (int) $website->getId();

                if (!$this->settings->isEnabled($websiteId)) {
                    continue;
                }

                $websiteName = $this->storeManager->getWebsite($websiteId)->getName();

                $itemCategories = $this->settings->getCategoryMapping($websiteId);
                if (null === $itemCategories) {
                    continue;
                }
                $this->erp->setWebsite($websiteId);
                foreach ($itemCategories as $category) {
                    $categoryFilter = $category['category_filter'];
                    $categoryId = $category['product_category'];
                    $attribSetId = $category['attribute_set'];

                    $this->logger->info('Retrieving products for category "' . $categoryFilter .
                                        '" (Website : ' . $websiteName . ')');

                    $json = $this->erp->getItemList($categoryFilter, $force);
                    $itemCollection->initItems($json, $websiteId, $categoryId, $categoryFilter, $attribSetId);

                    $json = $this->erp->getItemPrices($categoryFilter);
                    $itemCollection->initItemsPrices($json);

                    $json = $this->erp->getItemAttributes($categoryFilter);
                    $itemCollection->initItemsAttributes($json);
                }
                $this->erp->setItemImportCompleted();
            }
            if ($websiteName) {
                $this->erp->createCommonName($itemCollection);
                $this->createProducts($itemCollection);
            }
        } catch (\Throwable $exception) {
            $errorMessage = 'Website name : ' . $websiteName . "\n" . $errorMessage . $exception->getMessage() . "\n";
            $this->logger->error($errorMessage);
        }

        $this->isImportRunning = false;

        return $this->erp->postProcessing($errorMessage);
    }
}
