<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Settings;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;

class ConfigurableProduct extends ProductBase
{
    protected $productFactory;
    protected $productRepository;
    protected $productResourceModel;
    protected $settings;
    protected $stockRegistry;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        Settings $settings,
        ErpLogger $logger,
        ProductFactory $productFactory,
        Product $productResourceModel
    ) {
        parent::__construct(
            $productRepository,
            $settings,
            $logger
        );

        $this->productFactory = $productFactory;
        $this->productRepository = $productRepository;
        $this->productResourceModel = $productResourceModel;
        $this->settings = $settings;
    }

    public function createOrUpdateConfigurableProduct(\Ecomero\ErpCore\Model\Item $item): void
    {
        $configuarble_sku = $item->getCommmonName();

        if (null == $item->associatedProductIds) {
            $this->logger->error('  - No associated products found! (' . $configuarble_sku . ')');

            return;
        }

        if (1 == count($item->associatedProductIds)) {
            return;
        }

        try {
            /** @var \Magento\ConfigurableProduct\Model\Product\Type\Configurable $product */
            $product = $this->productRepository->get($configuarble_sku);
            $isNew = false;
            $this->logger->debug('  - Configurable product already exists, updating (' . $configuarble_sku . ')');
        } catch (NoSuchEntityException $e) {
            $product = $this->productFactory->create();
            $product->setSku($configuarble_sku);
            $product->setTypeId(Configurable::TYPE_CODE);
            $product->setVisibility(Visibility::VISIBILITY_BOTH);
            $product->setStatus(Status::STATUS_ENABLED);
            $isNew = true;
            $this->logger->debug('  - New configurable product must be created (' . $configuarble_sku . ')');
        }

        if (\Magento\ConfigurableProduct\Model\Product\Type\Configurable::TYPE_CODE !== $product->getTypeId()) {
            $this->logger->error('An existing simple product was found in Magento for ' . $item->getCommmonName() . ".\n" .
            "If you want to create a configurable product, then please delete the simple product in Magento and also in the ERP.\n" .
            'After the product is deleted, please re-run the import.');

            return;
        }
        /** @var \Magento\ConfigurableProduct\Model\Product\Type\Configurable $configurableProduct */
        $configurableProduct = $product->getTypeInstance();

        $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

        if ('' !== $isDirty) {
            $this->logger->debug('    ...found changes, saving product (' . $isDirty . ')');

            $product->setStockData(['use_config_manage_stock' => 1,
                'manage_stock' => 1,
                'is_qty_decimal' => 0,
                'is_in_stock' => 1, ]);

            // Super attribute
            $usedProductAttribArray = [];
            foreach ($item->attributes as $key => $value) {
                if (\Ecomero\ErpCore\Model\Attribute::YES === $value->usedInConfiguration) {
                    $attrib = $this->productResourceModel->getAttribute($key);
                    if ($attrib) {
                        $usedProductAttribArray[] = $attrib->getId();
                    }
                }
            }
            $configurableProduct->setUsedProductAttributeIds($usedProductAttribArray, $product);
            $configurableAttributesData = $configurableProduct->getConfigurableAttributesAsArray($product);
            $product->setCanSaveConfigurableAttributes(true);
            $product->setConfigurableAttributesData($configurableAttributesData);
            $configurableProductsData = [];
            $product->setConfigurableProductsData($configurableProductsData);

            $this->storeProductWithURLCheck($product, $item);
        }

        // Link simple products
        $usedProducts = $configurableProduct->getUsedProducts($product);
        $existingLinks = [];
        foreach ($usedProducts as $child) {
            $existingLinks[] = $child->getId();
        }

        // Only update if links are changed
        array_multisort($existingLinks);
        array_multisort($item->associatedProductIds);
        if (serialize($existingLinks) !== serialize($item->associatedProductIds)) {
            // Add missing products
            foreach ($usedProducts as $child) {
                if (!in_array($child->getId(), $item->associatedProductIds)) {
                    $item->associatedProductIds[] = $child->getId();
                }
            }

            $this->logger->debug('    ...found link changes, saving product');

            /** @var \Magento\Catalog\Model\Product $configurableProduct */
            $configurableProduct = $this->productRepository->get($configuarble_sku);
            $configurableProduct->setAssociatedProductIds($item->associatedProductIds);
            $configurableProduct->setCanSaveConfigurableAttributes(true);
            $this->storeProductWithURLCheck($product, $item);
        }
    }
}
