<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Controller\Index;

use Ecomero\PunchOut\Helper\Data;
use Ecomero\PunchOut\Helper\Logger;
use Ecomero\PunchOut\Model\OrderMessage;
use Ecomero\PunchOut\Model\PingMessage;
use Ecomero\PunchOut\Model\ProfileMessage;
use Ecomero\PunchOut\Model\SetupMessage;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\RawFactory;
use Magento\Framework\View\Result\PageFactory;
use Magento\Store\Model\StoreManagerInterface;

class Index extends \Magento\Framework\App\Action\Action implements CsrfAwareActionInterface
{
    protected $resultRawFactory;
    protected $storeManager;
    protected $orderMessage;
    protected $setupMessage;
    protected $profileMessage;
    protected $pingMessage;
    protected $logger;
    protected $data;

    public function __construct(
        Context $context,
        RawFactory  $resultRawFactor,
        PageFactory $resultPageFactory,
        StoreManagerInterface $storeManager,
        OrderMessage $orderMessage,
        SetupMessage $setupMessage,
        ProfileMessage $profileMessage,
        PingMessage $pingMessage,
        Logger $logger,
        Data $data
    ) {
        $this->resultRawFactory = $resultRawFactor;
        $this->resultPageFactory = $resultPageFactory;
        $this->storeManager = $storeManager;
        $this->orderMessage = $orderMessage;
        $this->setupMessage = $setupMessage;
        $this->profileMessage = $profileMessage;
        $this->pingMessage = $pingMessage;
        $this->logger = $logger;
        $this->data = $data;
        parent::__construct($context);
    }

    public function createCsrfValidationException(RequestInterface $request): ? InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    private function processMessage(
        string $body,
        \Magento\Framework\Controller\Result\Raw $result
    ) : \Magento\Framework\Controller\Result\Raw {
        $xml = new \SimpleXMLElement($body);
        $baseUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_WEB);

        if (isset($xml->Request->ProfileRequest)) {
            $rc = $this->profileMessage->processProfileRequest($xml, $baseUrl);
            $result->setStatusHeader($rc['code']);
        } elseif (isset($xml->Request->PunchOutSetupRequest)) {
            $rc = $this->setupMessage->processSetupRequest($xml, $baseUrl);
            $result->setStatusHeader($rc['code']);
        } elseif (isset($xml->Request->OrderRequest)) {
            $rc = $this->orderMessage->processOrderRequest($xml);
            $result->setStatusHeader($rc['code']);
        }

        $result->setContents($rc['xml']->asXML());
        return $result;
    }

    public function execute()
    {
        $result = $this->resultRawFactory->create();

        $result->setHeader('Cache-Control', 'no-cache', true);
        $result->setHeader('Content-Type', 'text/xml', true);
        $result->setHeader('Access-Control-Allow-Headers', '*', true);
        $result->setHeader('Access-Control-Allow-Origin', '*', true);

        try {
            if ($this->data->isEnabled('NO', Data::SCOPE_TYPE_STORE)) {
                $this->logger->error('ERROR: PunchOut is not enabled for this site');
                $result->setContents('PunchOut is not enabled for this site');
                $result->setStatusHeader(500);
            } elseif ($this->data->isEnabled('cXML', Data::SCOPE_TYPE_STORE)) {
                if ($this->getRequest()->isPost()) {
                    $body = $this->getRequest()->getContent();
                    $this->logger->info($body);
                    $result = $this->processMessage($body, $result);
                } else {
                    if ($this->getRequest()->getParam('token')) {
                        $this->logger->info('Request with token ' . $this->getRequest()->getParam('token'));
                        $page = $this->resultPageFactory->create();
                        return $page;
                    }

                    $this->logger->info('Ping request');
                    $rc = $this->pingMessage->processPingRequest();
                    $result->setStatusHeader($rc['code']);
                    $result->setContents($rc['xml']->asXML());
                }
            } elseif ($this->data->isEnabled('OCI', Data::SCOPE_TYPE_STORE)) {
                $hookUrl = $this->getRequest()->getParam('HOOK_URL');
                if (!$hookUrl) {
                    $this->logger->error('ERROR: Invalid PunchOut request');
                    $result->setContents('OCI RoundTrip require a HOOK_URL parameter');
                    $result->setStatusHeader(500);
                    return $result;
                } else {
                    $this->logger->info('OCI RoundTrip login request for hook : ' . $hookUrl);

                    if ($this->getRequest()->isPost()) {
                        $body = $this->getRequest()->getContent();
                        $this->logger->info($body);
                    }
                }
                $page = $this->resultPageFactory->create();
                return $page;
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $result->setContents('Internal error, see log for more information');
            $result->setStatusHeader(400);
        } catch (Throwable $t) {
            $this->logger->error($t->getMessage());
            $result->setContents('Internal error, see log for more information');
            $result->setStatusHeader(500);
        }

        return $result;
    }
}
