<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Model;

use Ecomero\PunchOut\Api\Data\SetupInterface;
use Ecomero\PunchOut\Model\ResourceModel\Setup\CollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class SetupRepository implements \Ecomero\PunchOut\Api\SetupRepositoryInterface
{
    private $setupFactory;
    private $setupItemFactory;
    private $collectionFactory;
    private $searchResultsFactory;
    private $collectionProcessor;
    private $setupItemRepository;
    private $searchCriteriaBuilder;

    public function __construct(
        SetupFactory $setupFactory,
        SetupItemFactory $setupItemFactory,
        CollectionFactory $collectionFactory,
        SearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor,
        SetupItemRepository $setupItemRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        $this->setupFactory = $setupFactory;
        $this->setupItemFactory = $setupItemFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->setupItemRepository = $setupItemRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    public function loadFromXml(\SimpleXMLElement $xml) : SetupInterface
    {
        $setup = $this->setupFactory->create();

        $po = $xml->Request->PunchOutSetupRequest;

        $setup->setPayloadId((string)$xml['payloadID']);
        $setup->setTimeStamp(new \DateTime((string)$xml['timestamp']));
        $setup->setHeaderFromDomain((string)$xml->Header->From->Credential['domain']);
        $setup->setHeaderFromIdentity((string)$xml->Header->From->Credential->Identity);
        $setup->setHeaderToDomain((string)$xml->Header->To->Credential['domain']);
        $setup->setHeaderToIdentity((string)$xml->Header->To->Credential->Identity);
        $setup->setHeaderSenderDomain((string)$xml->Header->Sender->Credential['domain']);
        $setup->setHeaderSenderIdentity((string)$xml->Header->Sender->Credential->Identity);
        $setup->setHeaderSenderUserAgent((string)$xml->Header->Sender->UserAgent);
        $setup->setRequestDeploymentMode((string)$xml->Request['deploymentMode']);
        $setup->setRequestOperation((string)$po['operation']);
        $setup->setRequestBuyerCookie((string)$po->BuyerCookie);
        $setup->setRequestBrowserFormPostUrl((string)$po->BrowserFormPost->URL);
        $setup->setRequestSupplierSetupUrl((string)$po->SupplierSetup->URL);

        $extrinsic = [];
        foreach ($xml->Request->PunchOutSetupRequest->Extrinsic as $value) {
            $extrinsic[ (string)$value['name']] = (string)$value;
        }
        $setup->setRequestExtrinsic(serialize($extrinsic));

        if (isset($po->Contact->Name)) {
            $setup->setRequestContactName((string)$po->Contact->Name);
        }
        if (isset($po->Contact->Email)) {
            $setup->setRequestContactEmail((string)$po->Contact->Email);
        }

        if (isset($po->ShipTo->Address)) {
            $setup->setRequestShipToId((string)$po->ShipTo->Address['addressID']);
            $setup->setRequestShipToName((string)$po->ShipTo->Address->Name);
            $setup->setRequestShipToStreet((string)$po->ShipTo->Address->PostalAddress->Street);
            $setup->setRequestShipToCity((string)$po->ShipTo->Address->PostalAddress->City);
            $setup->setRequestShipToState((string)$po->ShipTo->Address->PostalAddress->State);
            $setup->setRequestShipToPostalCode((string)$po->ShipTo->Address->PostalAddress->PostalCode);
            $setup->setRequestShipToCountry((string)$po->ShipTo->Address->PostalAddress->Country);
        }

        $items = [];
        if (isset($po->ItemOut)) {
            foreach ($po->ItemOut as $item) {
                $setupItem = $this->setupItemRepository->loadFromXml($item);
                $items [] = $setupItem;
            }
        }
        $setup->setItems($items);

        return $setup;
    }

    public function save(SetupInterface $setup) : SetupInterface
    {
        try {
            $setup->getResource()->save($setup);
            $parentId = (int)$setup->getId();
            foreach ($setup->getItems() as $item) {
                $item->setSetupId($parentId);
                $item->getResource()->save($item);
            }
        } catch (\Exception $e) {
            throw new CouldNotSaveException(__($e->getMessage()));
        }
        return $setup;
    }

    public function getById(string $setupId) : SetupInterface
    {
        $setup = $this->setupFactory->create();
        $setup->getResource()->load($setup, $setupId);
        if (!$setup->getId()) {
            throw new NoSuchEntityException(__('Setup with id "%1" does not exist', $setupId));
        }
        return $setup;
    }

    public function getByToken(string $token) : ?SetupInterface
    {
        $this->searchCriteriaBuilder->addFilter('token', $token);
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $oldSetupRequests  = $this->getList($searchCriteria)->getItems();
        foreach ($oldSetupRequests as $value) {
            return $value;
        }
        return null;
    }

    public function delete(SetupInterface $setup) : SetupInterface
    {
        try {
            $setup->getResource()->delete($setup);
        } catch (\Exception $e) {
            throw new CouldNotDeleteException(__($e->getMessage()));
        }
        return $setup;
    }

    public function deleteById(string $setupId) : SetupInterface
    {
        return $this->delete($this->getById($setupId));
    }

    public function getList(SearchCriteriaInterface $criteria) : SearchResultsInterface
    {
        $searchResult = $this->searchResultsFactory->create();
        $collection = $this->collectionFactory->create();
        $this->collectionProcessor->process($criteria, $collection);
        foreach ($collection->getItems() as $setupMessage) {
            $this->searchCriteriaBuilder->addFilter('setup_id', $setupMessage->getId());
            $searchCriteria = $this->searchCriteriaBuilder->create();
            $items  = $this->setupItemRepository->getList($searchCriteria)->getItems();
            $setupMessage->setItems($items);
        }
        $searchResult->setItems($collection->getItems());
        $searchResult->setTotalCount($collection->getSize());

        return $searchResult;
    }
}
