<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Model;

use Ecomero\PunchOut\Helper\Data;
use Ecomero\PunchOut\Helper\OrderHelper;
use Magento\Catalog\Model\ProductRepository;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\Quote\Model\Quote;

class OrderMessage extends MessageBase
{
    protected $productRepository;
    protected $configurableProduct;
    protected $data;
    protected $setupItemRepository;
    protected $orderHelper;

    public function __construct(
        SetupItemRepository $setupItemRepository,
        OrderHelper $orderHelper,
        ProductRepository $productRepository,
        Configurable $configurableProduct,
        Data $data
    ) {
        $this->setupItemRepository = $setupItemRepository;
        $this->orderHelper = $orderHelper;
        $this->productRepository = $productRepository;
        $this->configurableProduct = $configurableProduct;
        $this->data = $data;
        parent::__construct($data);
    }

    private function getAddress(\SimpleXMLElement $xml) : array
    {
        return [
            'street' => (string)$xml->Address->PostalAddress->Street,
            'city' => (string)$xml->Address->PostalAddress->City,
            'country_id' => (string)$xml->Address->PostalAddress->Country['isoCountryCode'],
            'region' => (string)$xml->Address->PostalAddress->State,
            'postcode' => (string)$xml->Address->PostalAddress->PostalCode
        ];
    }

    public function processOrderRequest(\SimpleXMLElement $xml) : array
    {
        if (!$this->isAuthorized($xml)) {
            $xmlResponse = $this->createXmlResponse((string)$xml['payloadID'], 401, 'Unauthorized');
            return ['code' => 401,
                    'xml' => $xmlResponse ];
        }

        if ((string)$xml->Request->OrderRequest->OrderRequestHeader['type'] !== 'new') {
            $xmlResponse = $this->createXmlResponse((string)$xml['payloadID'], 417, 'Expectation Failed');
            return ['code' => 417,
                    'xml' => $xmlResponse ];
        }

        $email = (string)$xml->Request->OrderRequest->OrderRequestHeader->Contact->Email;

        $items = [];
        if (isset($xml->Request->OrderRequest->ItemOut)) {
            foreach ($xml->Request->OrderRequest->ItemOut as $item) {
                $setupItem = $this->setupItemRepository->loadFromXml($item);
                $items [] = $setupItem;
            }
        }

        $shippingAddress = $this->getAddress($xml->Request->OrderRequest->OrderRequestHeader->ShipTo);
        $billingAddress = $this->getAddress($xml->Request->OrderRequest->OrderRequestHeader->BillTo);

        $this->orderHelper->createOrder($email, $items, $shippingAddress, $billingAddress);

        $xmlResponse = $this->createXmlResponse((string)$xml['payloadID'], 200, 'OK');
        return ['code' => 200,
                'xml' => $xmlResponse ];
    }

    private function createHeader(\SimpleXMLElement $xml) : \SimpleXMLElement
    {
        $headerNode = $xml->addChild('Header');
        $fromNode = $headerNode->addChild('From');
        $fromCredentialNode = $fromNode->addChild('Credential');
        $fromCredentialNode->addAttribute('domain', '');
        $fromCredentialIdentityNode = $fromCredentialNode->addChild('Identity');

        $toNode = $headerNode->addChild('To');
        $toCredentialNode = $toNode->addChild('Credential');
        $toCredentialNode->addAttribute('domain', '');
        $toCredentialIdentityNode = $toCredentialNode->addChild('Identity');

        $senderNode = $headerNode->addChild('Sender');
        $senderCredentialNode = $senderNode->addChild('Credential');
        $senderCredentialNode->addAttribute('domain', '');
        $senderCredentialIdentityNode = $senderCredentialNode->addChild('Identity');
        $senderNode->addChild('UserAgent', 'Magento eComero Punchout');

        return $xml;
    }

    public function addItem(
        \Magento\Quote\Model\Quote\Item $item,
        \SimpleXMLElement $node,
        string $currencyCode
    ) : void {
        $itemNode = $node->addChild('ItemIn');
        $itemNode->addAttribute('quantity', strval($item->getTotalQty()));

        $itemIdNode = $itemNode->addChild('ItemID');
        $itemIdNode->addChild('SupplierPartID', $item->getSku());

        $product = $this->productRepository->get($item->getSku());
        $productId = $product->getId();
        $parentIds = $this->configurableProduct->getParentIdsByChild($productId);
        if (isset($parentIds[0])) {
            $productId = $parentIds[0];
        }

        $itemIdNode->addChild('SupplierPartAuxiliaryID', $productId);

        $itemDetailNode = $itemNode->addChild('ItemDetail');
        $unitPriceNode = $itemDetailNode->addChild('UnitPrice');
        $moneyNode = $unitPriceNode->addChild('Money', strval($item->getPrice()));
        $moneyNode->addAttribute('currency', $currencyCode);

        $descriptionNode = $itemDetailNode->addChild('Description', $item->getDescription());
        $descriptionNode->addAttribute('xml:lang', 'en');
        $descriptionNode->ShortName = $item->getName();
        $attrib = $product->getCustomAttribute($this->data->getUnitOfMeasure(Data::SCOPE_TYPE_STORE));
        if ($attrib !== null) {
            $unitOfMeasure = $attrib->getValue();
            $itemDetailNode->addChild('UnitOfMeasure', $unitOfMeasure);
        } else {
            $itemDetailNode->addChild('UnitOfMeasure', 'EA');
        }

        $code = '';
        $attrib = $product->getCustomAttribute($this->data->getClassificationCode(Data::SCOPE_TYPE_STORE));
        if ($attrib !== null) {
            $code = $attrib->getValue();
        }
        $itemDetailNode->addChild('Classification', $code)
                        ->addAttribute('domain', $this->data->getClassificationDomain(Data::SCOPE_TYPE_STORE));
    }

    public function createOrderFromCart(
        Quote $quote,
        string $currencyCode,
        string $payloadId,
        string $buyerCookie
    ) : string {
        $xml_header =   '<?xml version="1.0" encoding="UTF-8"?>' .
        '<!DOCTYPE cXML SYSTEM "http://xml.cXML.org/schemas/cXML/1.2.040/cXML.dtd"><cXML></cXML>';
        $xml = new \SimpleXMLElement($xml_header);
        $xml->addAttribute('payloadID', $payloadId);
        $now = new \DateTime();
        $xml->addAttribute('timestamp', $now->format('Y-m-d\TH:i:sP'));

        $this->createHeader($xml);
        $messageNode = $xml->addChild('Message');
        $orderMessageNode = $messageNode->addChild('PunchOutOrderMessage');
        $orderMessageNode->addChild('BuyerCookie', $buyerCookie);
        $orderMessageHeaderNode = $orderMessageNode->addChild('PunchOutOrderMessageHeader');
        $orderMessageHeaderNode->addAttribute('operationAllowed', 'edit');
        $totalNode = $orderMessageHeaderNode->addChild('Total');
        $moneyNode = $totalNode->addChild('Money', $quote->getSubtotal());

        $moneyNode->addAttribute('currency', $currencyCode);

        foreach ($quote->getAllVisibleItems() as $item) {
            $this->addItem($item, $orderMessageNode, $currencyCode);
        }

        return $xml->asXML();
    }
}
