<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Model\Proceedo;

use Ecomero\PunchOut\Helper\Data;
use Ecomero\PunchOut\Helper\OrderHelper;
use Ecomero\PunchOut\Model\SetupItemRepository;
use Magento\Catalog\Model\ProductRepository;
use Magento\Catalog\Model\ResourceModel\Product as ProductResourceModel;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Quote\Model\Quote;
use Magento\Store\Model\Information;
use Magento\Store\Model\StoreManagerInterface;

class OrderMessage extends \Ecomero\PunchOut\Model\MessageBase
{
    protected $setupItemRepository;
    protected $productRepository;
    protected $productResourceModel;
    protected $arrayManager;
    protected $configurableProduct;
    protected $data;
    protected $orderHelper;
    protected $storeInformation;
    protected $storeManager;
    protected $customerSession;

    public function __construct(
        SetupItemRepository $setupItemRepository,
        OrderHelper $orderHelper,
        ProductRepository $productRepository,
        ProductResourceModel $productResourceModel,
        ArrayManager $arrayManager,
        Configurable $configurableProduct,
        Data $data,
        Information $storeInformation,
        StoreManagerInterface $storeManager,
        CustomerSession $customerSession
    ) {
        $this->setupItemRepository = $setupItemRepository;
        $this->orderHelper = $orderHelper;
        $this->productRepository = $productRepository;
        $this->productResourceModel = $productResourceModel;
        $this->arrayManager = $arrayManager;
        $this->configurableProduct = $configurableProduct;
        $this->data = $data;
        $this->storeInformation = $storeInformation;
        $this->storeManager = $storeManager;
        $this->customerSession = $customerSession;
        parent::__construct($data);
    }

    private function getAddress(\SimpleXMLElement $xml, array $name, string $telephone) : array
    {
        return [
            'firstname' => trim($this->arrayManager->get(0, $name) ?? '-'),
            'lastname' => trim($this->arrayManager->get(1, $name) ?? '-'),
            'street' => (string)$xml->Address->PostalAddress->Street ?: '-',
            'city' => (string)$xml->Address->PostalAddress->City,
            'country_id' => (string)$xml->Address->PostalAddress->Country['isoCountryCode'],
            'region' => (string)$xml->Address->PostalAddress->State,
            'postcode' => (string)$xml->Address->PostalAddress->PostalCode,
            'telephone' => $telephone
        ];
    }

    public function processOrderRequest(\SimpleXMLElement $xml) : array
    {
        $email = (string)$xml->Request->OrderRequest->OrderRequestHeader->Contact->Email;
        $telephone = (string)$xml->Request->OrderRequest->OrderRequestHeader->Contact->Phone->TelephoneNumber->Number;
        $name = explode(',', (string)$xml->Request->OrderRequest->OrderRequestHeader->Contact->Name ?? '');

        $items = [];
        if (isset($xml->Request->OrderRequest->ItemOut)) {
            foreach ($xml->Request->OrderRequest->ItemOut as $item) {
                $setupItem = $this->setupItemRepository->loadFromXml($item);
                $items [] = $setupItem;
            }
        }

        $shippingAddress = $this->getAddress($xml->Request->OrderRequest->OrderRequestHeader->ShipTo, $name, $telephone);
        $billingAddress = $this->getAddress($xml->Request->OrderRequest->OrderRequestHeader->BillTo, $name, $telephone);

        $order = $this->orderHelper->createOrder($email, $items, $shippingAddress, $billingAddress);
        $header = $xml->Request->OrderRequest->OrderRequestHeader;
        $externOrderId = '';
        $agreementId = '';
        if ($header instanceof \SimpleXMLElement) {
            $externOrderId = (string)$header->attributes()->orderID;
            $agreementId = (string)$header->attributes()->agreementID;
        }
        $order->addCommentToStatusHistory("Order created via cXML, payload: {$xml['payloadID']}, order id: {$externOrderId}, agreement id: {$agreementId}");
        $order->save();

        $xmlResponse = $this->createXmlResponse((string)$xml['payloadID'], 200, 'OK');
        return ['code' => 200,
                'xml' => $xmlResponse ];
    }

    private function getStoreVatNumber() : string
    {
        $store = $this->storeManager->getStore();
        $info = $this->storeInformation->getStoreInformationObject($store);
        return $info->getVatNumber() ?? '';
    }

    private function createHeader(
        \SimpleXMLElement $xml,
        string $prosupplierid,
        string $prosuppliername
    ) : \SimpleXMLElement {
        $headerNode = $xml->addChild('Header');
        $fromNode = $headerNode->addChild('From');
        $fromCredentialNode = $fromNode->addChild('Credential');
        $fromCredentialNode->addAttribute('domain', 'PROSUPPLIERNAME');
        $fromCredentialIdentityNode = $fromCredentialNode->addChild('Identity', $prosuppliername);
        $fromCredentialNode = $fromNode->addChild('Credential');
        $fromCredentialNode->addAttribute('domain', 'PROSUPPLIERID');
        $fromCredentialIdentityNode = $fromCredentialNode->addChild('Identity', $prosupplierid);

        $toNode = $headerNode->addChild('To');
        $toCredentialNode = $toNode->addChild('Credential');
        $toCredentialNode->addAttribute('domain', 'VATNO');
        $toCredentialIdentityNode = $toCredentialNode->addChild('Identity', $this->customerSession->getCustomer()->getTaxvat());
        $toCredentialNode = $toNode->addChild('Credential');
        $toCredentialNode->addAttribute('domain', 'NAME');
        $toCredentialIdentityNode = $toCredentialNode->addChild('Identity', $this->customerSession->getCustomer()->getName());

        $senderNode = $headerNode->addChild('Sender');
        $senderCredentialNode = $senderNode->addChild('Credential');
        $senderCredentialNode->addAttribute('domain', 'VATNO');
        $senderCredentialIdentityNode = $senderCredentialNode->addChild('Identity', $this->getStoreVatNumber());

        $senderNode->addChild('UserAgent');

        return $xml;
    }

    private function addTaxItem(
        \Magento\Quote\Model\Quote\Item $item,
        \SimpleXMLElement $itemNode,
        string $currencyCode
    ) {
        $taxNode = $itemNode->addChild('Tax');
        $moneyNode = $taxNode->addChild('Money', number_format((float)$item->getTaxAmount(), 2, '.', ''));
        $moneyNode->addAttribute('currency', $currencyCode);
        $descriptionNode = $taxNode->addChild('Description', 'VAT');
        $descriptionNode->addAttribute('xml:xml:lang', 'SV');
        $taxDetailNode = $taxNode->addChild('TaxDetail');
        $taxDetailNode->addAttribute('category', 'VAT');
        $taxDetailNode->addAttribute('percentageRate', number_format((float)$item->getTaxPercent(), 0));
        $taxAmountlNode = $taxDetailNode->addChild('TaxAmount');
        $moneyNode = $taxAmountlNode->addChild('Money', number_format((float)$item->getTaxAmount(), 2, '.', ''));
        $moneyNode->addAttribute('currency', $currencyCode);
    }

    private function addItem(
        \Magento\Quote\Model\Quote\Item $item,
        \SimpleXMLElement $node,
        string $currencyCode
    ) : void {
        $itemNode = $node->addChild('ItemIn');
        $itemNode->addAttribute('quantity', strval($item->getTotalQty()));

        $itemIdNode = $itemNode->addChild('ItemID');
        $itemIdNode->addChild('SupplierPartID', $item->getSku());

        $product = $this->productRepository->get($item->getSku());
        $productId = $product->getId();
        $parentIds = $this->configurableProduct->getParentIdsByChild($productId);
        if (isset($parentIds[0])) {
            $productId = $parentIds[0];
        }

        $itemIdNode->addChild('SupplierPartAuxiliaryID', $productId);

        $itemDetailNode = $itemNode->addChild('ItemDetail');
        $unitPriceNode = $itemDetailNode->addChild('UnitPrice');
        $moneyNode = $unitPriceNode->addChild('Money', number_format((float)$item->getPrice(), 2, '.', ''));
        $moneyNode->addAttribute('currency', $currencyCode);

        $descriptionNode = $itemDetailNode->addChild('Description', strip_tags($product->getDescription() ?? ''));
        $descriptionNode->addAttribute('xml:xml:lang', 'SV');
        $descriptionNode->ShortName = $item->getName();

        $prodAttrib = $product->getCustomAttribute($this->data->getUnitOfMeasure(Data::SCOPE_TYPE_STORE));
        $eavAttrib = $this->productResourceModel->getAttribute($this->data->getUnitOfMeasure(Data::SCOPE_TYPE_STORE));

        if ($prodAttrib && $eavAttrib) {
            if ($eavAttrib->usesSource()) {
                $optionId = $prodAttrib->getValue();
                $unitOfMeasure = $eavAttrib->getSource()->getOptionText($optionId);
                $itemDetailNode->addChild('UnitOfMeasure', $unitOfMeasure);
            } else {
                $itemDetailNode->addChild('UnitOfMeasure', $prodAttrib->getValue());
            }
        } else {
            $itemDetailNode->addChild('UnitOfMeasure', 'EA');
        }

        $this->addTaxItem($item, $itemNode, $currencyCode);

        $code = '';
        $attrib = $product->getCustomAttribute($this->data->getClassificationCode(Data::SCOPE_TYPE_STORE));
        if ($attrib !== null) {
            $code = $attrib->getValue();
        }
        $itemDetailNode->addChild('Classification', $code)
                        ->addAttribute('domain', $this->data->getClassificationDomain(Data::SCOPE_TYPE_STORE));
    }

    public function createOrderFromCart(
        Quote $quote,
        string $currencyCode,
        string $buyerCookie,
        string $prosupplierid,
        string $prosuppliername
    ) : string {
        $xml_header =   '<?xml version="1.0" encoding="UTF-8"?>' .
        '<!DOCTYPE cXML SYSTEM "http://xml.cXML.org/schemas/cXML/1.2.012/cXML.dtd"><cXML></cXML>';
        $xml = new \SimpleXMLElement($xml_header);
        $xml->addAttribute('payloadID', md5($this->createPayloadId()));
        $now = new \DateTime();
        $xml->addAttribute('timestamp', $now->format('Y-m-d\TH:i:sP'));
        $xml->addAttribute('xml:xml:lang', 'SV');
        $this->createHeader($xml, $prosupplierid, $prosuppliername);

        $messageNode = $xml->addChild('Message');
        $orderMessageNode = $messageNode->addChild('PunchOutOrderMessage');
        $orderMessageNode->addChild('BuyerCookie', $buyerCookie);
        $orderMessageHeaderNode = $orderMessageNode->addChild('PunchOutOrderMessageHeader');
        $orderMessageHeaderNode->addAttribute('operationAllowed', 'create');

        $totalNode = $orderMessageHeaderNode->addChild('Total');
        $moneyNode = $totalNode->addChild('Money', number_format((float)$quote->getSubtotal(), 2, '.', ''));
        $moneyNode->addAttribute('currency', $currencyCode);

        $taxNode = $orderMessageHeaderNode->addChild('Tax');
        $moneyNode = $taxNode->addChild('Money', number_format((float)$quote->getShippingAddress()->getTaxAmount(), 2, '.', ''));
        $moneyNode->addAttribute('currency', $currencyCode);
        $descriptionNode = $taxNode->addChild('Description', 'VAT');
        $descriptionNode->addAttribute('xml:xml:lang', 'SV');

        foreach ($quote->getAllVisibleItems() as $item) {
            $this->addItem($item, $orderMessageNode, $currencyCode);
        }

        return $xml->asXML();
    }
}
