<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Api\Data;

/**
 * PunchOut Setup Interface
 *
 * Represents a PunchOut setup request containing session and configuration data
 * for cXML PunchOut transactions
 *
 * @api
 * @since 2.0.0
 */
interface SetupInterface
{
    /** @var string Database column name for setup ID */
    public const SETUP_ID = 'setup_id';
    /** @var string Database column name for session token */
    public const TOKEN = 'token';
    /** @var string Database column name for session ID */
    public const SESSION_ID = 'session_id';
    /** @var string Database column name for creation timestamp */
    public const CREATED = 'created';
    /** @var string Database column name for cXML payload ID */
    public const PAYLOAD_ID = 'payload_id';
    /** @var string Database column name for request timestamp */
    public const TIMESTAMP = 'timestamp';
    /** @var string Database column name for sender domain */
    public const HEADER_FROM_DOMAIN = 'header_from_domain';
    /** @var string Database column name for sender identity */
    public const HEADER_FROM_IDENTITY = 'header_from_identity';
    /** @var string Database column name for recipient domain */
    public const HEADER_TO_DOMAIN = 'header_to_domain';
    /** @var string Database column name for recipient identity */
    public const HEADER_TO_IDENTITY = 'header_to_identity';
    /** @var string Database column name for sender domain */
    public const HEADER_SENDER_DOMAIN = 'header_sender_domain';
    /** @var string Database column name for sender identity */
    public const HEADER_SENDER_IDENTITY = 'header_sender_identity';
    /** @var string Database column name for sender user agent */
    public const HEADER_SENDER_USER_AGENT = 'header_sender_user_agent';
    /** @var string Database column name for deployment mode */
    public const REQUEST_DEPLOYMENT_MODE = 'req_deployment_mode';
    /** @var string Database column name for operation type */
    public const REQUEST_OPERATION = 'req_operation';
    /** @var string Database column name for buyer cookie */
    public const REQUEST_BUYER_COOKIE = 'req_buyer_cookie';
    /** @var string Database column name for return URL */
    public const REQUEST_BROWSER_FORM_POST_URL = 'req_browser_form_post_url';
    /** @var string Database column name for supplier setup URL */
    public const REQUEST_SUPPLIER_SETUP_URL = 'req_supplier_setup_url';
    /** @var string Database column name for extrinsic data */
    public const REQUEST_EXTRINSIC = 'req_extrinsic';
    /** @var string Database column name for contact name */
    public const REQUEST_CONTACT_NAME = 'req_contact_name';
    /** @var string Database column name for contact email */
    public const REQUEST_CONTACT_EMAIL = 'req_contact_email';
    /** @var string Database column name for ship-to address ID */
    public const REQUEST_SHIP_TO_ID = 'req_ship_to_id';
    /** @var string Database column name for ship-to name */
    public const REQUEST_SHIP_TO_NAME = 'req_ship_to_name';
    /** @var string Database column name for ship-to street */
    public const REQUEST_SHIP_TO_STREET = 'req_ship_to_street';
    /** @var string Database column name for ship-to city */
    public const REQUEST_SHIP_TO_CITY = 'req_ship_to_city';
    /** @var string Database column name for ship-to state */
    public const REQUEST_SHIP_TO_STATE = 'req_ship_to_state';
    /** @var string Database column name for ship-to postal code */
    public const REQUEST_SHIP_TO_POSTAL_CODE = 'req_ship_to_postal_code';
    /** @var string Database column name for ship-to country */
    public const REQUEST_SHIP_TO_COUNTRY = 'req_ship_to_country';

    /**
     * Get setup items collection
     *
     * @return \Ecomero\PunchOut\Api\Data\SetupItemInterface[]
     * @since 2.0.0
     */
    public function getItems() : array;

    /**
     * Set setup items collection
     *
     * @param \Ecomero\PunchOut\Api\Data\SetupItemInterface[] $items
     * @return void
     * @since 2.0.0
     */
    public function setItems(array $items);

    /**
     * Get session token
     *
     * @return string Unique session token for PunchOut session
     * @since 2.0.0
     */
    public function getToken() : string;

    /**
     * Set session token
     *
     * @param string $token Unique session token
     * @return void
     * @since 2.0.0
     */
    public function setToken(string $token) : void;

    /**
     * Get Magento session ID
     *
     * @return string|null Session ID when user accesses the session
     * @since 2.0.0
     */
    public function getSessionId() : ?string;

    /**
     * Set Magento session ID
     *
     * @param string $sessionId Active session ID
     * @return void
     * @since 2.0.0
     */
    public function setSessionId(string $sessionId) : void;

    /**
     * Get creation timestamp
     *
     * @return \DateTime|null When the setup was created
     * @since 2.0.0
     */
    public function getCreated() : ?\DateTime;

    /**
     * Set creation timestamp
     *
     * @param \DateTime $date Creation date and time
     * @return void
     * @since 2.0.0
     */
    public function setCreated(\DateTime $date) : void;

    /**
     * Get cXML payload ID
     *
     * @return string Unique payload identifier from cXML request
     * @since 2.0.0
     */
    public function getPayloadId() : string;

    /**
     * Set cXML payload ID
     *
     * @param string $payloadId Payload identifier
     * @return void
     * @since 2.0.0
     */
    public function setPayloadId(string $payloadId) : void;

    /**
     * Get request timestamp
     *
     * @return \DateTime|null When the setup request was sent
     * @since 2.0.0
     */
    public function getTimeStamp() : ?\DateTime;

    /**
     * Set request timestamp
     *
     * @param \DateTime $date Request date and time
     * @return void
     * @since 2.0.0
     */
    public function setTimeStamp(\DateTime $date) : void;

    /**
     * Get sender domain from cXML header
     *
     * @return string|null Sender's domain identifier
     * @since 2.0.0
     */
    public function getHeaderFromDomain() : ?string;

    /**
     * Set sender domain in cXML header
     *
     * @param string $domain Sender's domain
     * @return void
     * @since 2.0.0
     */
    public function setHeaderFromDomain(string $domain) : void;

    /**
     * Get sender identity from cXML header
     *
     * @return string|null Sender's identity
     * @since 2.0.0
     */
    public function getHeaderFromIdentity() : ?string;

    /**
     * Set sender identity in cXML header
     *
     * @param string $identity Sender's identity
     * @return void
     * @since 2.0.0
     */
    public function setHeaderFromIdentity(string $identity) : void;

    /**
     * Get recipient domain from cXML header
     *
     * @return string|null Recipient's domain identifier
     * @since 2.0.0
     */
    public function getHeaderToDomain() : ?string;

    /**
     * Set recipient domain in cXML header
     *
     * @param string $domain Recipient's domain
     * @return void
     * @since 2.0.0
     */
    public function setHeaderToDomain(string $domain) : void;

    /**
     * Get recipient identity from cXML header
     *
     * @return string|null Recipient's identity
     * @since 2.0.0
     */
    public function getHeaderToIdentity() : ?string;

    /**
     * Set recipient identity in cXML header
     *
     * @param string $identity Recipient's identity
     * @return void
     * @since 2.0.0
     */
    public function setHeaderToIdentity(string $identity) : void;

    /**
     * Get HTTP sender domain
     *
     * @return string|null HTTP request sender domain
     * @since 2.0.0
     */
    public function getHeaderSenderDomain() : ?string;

    /**
     * Set HTTP sender domain
     *
     * @param string $domain Sender domain
     * @return void
     * @since 2.0.0
     */
    public function setHeaderSenderDomain(string $domain) : void;

    /**
     * Get HTTP sender identity
     *
     * @return string Sender identity (organization/system)
     * @since 2.0.0
     */
    public function getHeaderSenderIdentity() : string;

    /**
     * Set HTTP sender identity
     *
     * @param string $identity Sender identity
     * @return void
     * @since 2.0.0
     */
    public function setHeaderSenderIdentity(string $identity) : void;

    /**
     * Get sender user agent
     *
     * @return string|null User agent string from HTTP request
     * @since 2.0.0
     */
    public function getHeaderSenderUserAgent() : ?string;

    /**
     * Set sender user agent
     *
     * @param string $userAgent User agent string
     * @return void
     * @since 2.0.0
     */
    public function setHeaderSenderUserAgent(string $userAgent) : void;

    /**
     * Get deployment mode
     *
     * @return string|null Deployment mode (production, test, etc.)
     * @since 2.0.0
     */
    public function getRequestDeploymentMode() : ?string;

    /**
     * Set deployment mode
     *
     * @param string $mode Deployment mode
     * @return void
     * @since 2.0.0
     */
    public function setRequestDeploymentMode(string $mode) : void;

    /**
     * Get PunchOut operation type
     *
     * @return string Operation type (create, edit, inspect)
     * @since 2.0.0
     */
    public function getRequestOperation() : string;

    /**
     * Set PunchOut operation type
     *
     * @param string $operation Operation type
     * @return void
     * @since 2.0.0
     */
    public function setRequestOperation(string $operation) : void;

    /**
     * Get buyer cookie
     *
     * @return string|null Cookie value for buyer identification
     * @since 2.0.0
     */
    public function getRequestBuyerCookie() : ?string;

    /**
     * Set buyer cookie
     *
     * @param string $cookie Cookie value
     * @return void
     * @since 2.0.0
     */
    public function setRequestBuyerCookie(string $cookie) : void;

    /**
     * Get return URL for cart submission
     *
     * @return string URL where cart should be posted after shopping
     * @since 2.0.0
     */
    public function getRequestBrowserFormPostUrl() : string;

    /**
     * Set return URL for cart submission
     *
     * @param string $url Return URL
     * @return void
     * @since 2.0.0
     */
    public function setRequestBrowserFormPostUrl(string $url) : void;

    /**
     * Get supplier setup URL
     *
     * @return string URL of this PunchOut catalog
     * @since 2.0.0
     */
    public function getRequestSupplierSetupUrl() : string;

    /**
     * Set supplier setup URL
     *
     * @param string $url Supplier URL
     * @return void
     * @since 2.0.0
     */
    public function setRequestSupplierSetupUrl(string $url) : void;

    /**
     * Get extrinsic data
     *
     * @return string|null Additional custom data passed in request
     * @since 2.0.0
     */
    public function getRequestExtrinsic() : ?string;

    /**
     * Set extrinsic data
     *
     * @param string $name Extrinsic data
     * @return void
     * @since 2.0.0
     */
    public function setRequestExtrinsic(string $name) : void;

    /**
     * Get contact person name
     *
     * @return string|null Name of requesting contact
     * @since 2.0.0
     */
    public function getRequestContactName() : ?string;

    /**
     * Set contact person name
     *
     * @param string $name Contact name
     * @return void
     * @since 2.0.0
     */
    public function setRequestContactName(string $name) : void;

    /**
     * Get contact email address
     *
     * @return string|null Email of requesting contact
     * @since 2.0.0
     */
    public function getRequestContactEmail() : ?string;

    /**
     * Set contact email address
     *
     * @param string $name Contact email
     * @return void
     * @since 2.0.0
     */
    public function setRequestContactEmail(string $name) : void;

    /**
     * Get ship-to address ID
     *
     * @return string|null Shipping address identifier
     * @since 2.0.0
     */
    public function getRequestShipToId() : ?string;

    /**
     * Set ship-to address ID
     *
     * @param string $id Address ID
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToId(string $id) : void;

    /**
     * Get ship-to recipient name
     *
     * @return string|null Shipping recipient name
     * @since 2.0.0
     */
    public function getRequestShipToName() : ?string;

    /**
     * Set ship-to recipient name
     *
     * @param string $name Recipient name
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToName(string $name) : void;

    /**
     * Get ship-to street address
     *
     * @return string|null Shipping street address
     * @since 2.0.0
     */
    public function getRequestShipToStreet() : ?string;

    /**
     * Set ship-to street address
     *
     * @param string $street Street address
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToStreet(string $street) : void;

    /**
     * Get ship-to city
     *
     * @return string|null Shipping city
     * @since 2.0.0
     */
    public function getRequestShipToCity() : ?string;

    /**
     * Set ship-to city
     *
     * @param string $city City name
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToCity(string $city) : void;

    /**
     * Get ship-to state/province
     *
     * @return string|null Shipping state or province
     * @since 2.0.0
     */
    public function getRequestShipToState() : ?string;

    /**
     * Set ship-to state/province
     *
     * @param string $state State or province
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToState(string $state) : void;

    /**
     * Get ship-to postal code
     *
     * @return string|null Shipping postal/ZIP code
     * @since 2.0.0
     */
    public function getRequestShipToPostalCode() : ?string;

    /**
     * Set ship-to postal code
     *
     * @param string $postalCode Postal/ZIP code
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToPostalCode(string $postalCode) : void;

    /**
     * Get ship-to country
     *
     * @return string|null Shipping country code
     * @since 2.0.0
     */
    public function getRequestShipToCountry() : ?string;

    /**
     * Set ship-to country
     *
     * @param string $country Country code (e.g., US, SE, DE)
     * @return void
     * @since 2.0.0
     */
    public function setRequestShipToCountry(string $country) : void;
}
