<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Api;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * PunchOut Setup Repository Interface
 *
 * Service contract for managing PunchOut setup records
 *
 * @api
 * @since 2.0.0
 */
interface SetupRepositoryInterface
{
    /**
     * Load setup from cXML
     *
     * Parses a cXML setup request and creates a Setup entity with all
     * header, request, and item data extracted from the XML structure
     *
     * @param \SimpleXMLElement $xml The cXML setup request document
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface
     * @throws \InvalidArgumentException If XML structure is invalid
     * @since 2.0.0
     */
    public function loadFromXml(\SimpleXMLElement $xml) : Data\SetupInterface;

    /**
     * Save setup
     *
     * Persists a setup record and all associated setup items to the database
     *
     * @param \Ecomero\PunchOut\Api\Data\SetupInterface $setup Setup to save
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface Saved setup with generated ID
     * @throws CouldNotSaveException If save operation fails
     * @since 2.0.0
     */
    public function save(Data\SetupInterface $setup) : Data\SetupInterface;

    /**
     * Get setup by ID
     *
     * Retrieves a setup record by its database ID
     *
     * @param string $setupId Setup database ID
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface
     * @throws NoSuchEntityException If setup with given ID doesn't exist
     * @since 2.0.0
     */
    public function getById(string $setupId) : Data\SetupInterface;

    /**
     * Get setup by token
     *
     * Retrieves a setup record by its unique session token.
     * Returns null if token is not found or has expired.
     *
     * @param string $token Unique session token
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface|null
     * @since 2.0.0
     */
    public function getByToken(string $token) : ?Data\SetupInterface;

    /**
     * Delete setup
     *
     * Removes a setup record and all associated items from database
     *
     * @param \Ecomero\PunchOut\Api\Data\SetupInterface $setup Setup to delete
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface Deleted setup
     * @throws CouldNotDeleteException If delete operation fails
     * @since 2.0.0
     */
    public function delete(Data\SetupInterface $setup) : Data\SetupInterface;

    /**
     * Delete setup by ID
     *
     * Removes a setup record by its database ID
     *
     * @param string $setupId Setup database ID
     * @return \Ecomero\PunchOut\Api\Data\SetupInterface Deleted setup
     * @throws NoSuchEntityException If setup doesn't exist
     * @throws CouldNotDeleteException If delete operation fails
     * @since 2.0.0
     */
    public function deleteById(string $setupId) : Data\SetupInterface;

    /**
     * Get list of setups matching search criteria
     *
     * Returns a collection of setup records based on provided search criteria.
     * Useful for filtering, sorting, and pagination of setup records.
     *
     * @param SearchCriteriaInterface $criteria Search criteria for filtering
     * @return SearchResultsInterface Search results containing matching setups
     * @throws \Magento\Framework\Exception\LocalizedException
     * @since 2.0.0
     */
    public function getList(SearchCriteriaInterface $criteria) : SearchResultsInterface;
}
