<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\PunchOut\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\Information;
use Magento\Store\Model\StoreManagerInterface;

class Data extends AbstractHelper
{
    public const SCOPE_TYPE_STORE = 'store';
    protected const MAX_PUNCHOUT_SESSION_LENGTH = (60 * 60);

    public function __construct(
        Context $context,
        protected EncryptorInterface $encryptor,
        protected Information $storeInformation,
        protected StoreManagerInterface $storeManager
    ) {
        parent::__construct($context);
    }

    public function isEnabled(string $method, string $scope = Data::SCOPE_TYPE_STORE): bool
    {
        return $this->scopeConfig->getValue('punchout/connection/enabled', $scope) === $method;
    }

    public function isLoginUser(string $scope = Data::SCOPE_TYPE_STORE): bool
    {
        return '1' === $this->scopeConfig->getValue('punchout/connection/login_user', $scope);
    }

    public function isLockedDown(string $scope = Data::SCOPE_TYPE_STORE): bool
    {
        return '1' === $this->scopeConfig->getValue('punchout/connection/lockdown', $scope);
    }

    public function isAllowStoreChange(string $scope = Data::SCOPE_TYPE_STORE): bool
    {
        return '1' === $this->scopeConfig->getValue('punchout/connection/store_change', $scope);
    }

    public function getSessionTimeout(string $scope = Data::SCOPE_TYPE_STORE): int
    {
        $timeout = $this->scopeConfig->getValue('punchout/connection/session_timeout', $scope);

        return null === $timeout ? self::MAX_PUNCHOUT_SESSION_LENGTH : (int) $timeout;
    }

    public function getSenderIdentity(string $scope = Data::SCOPE_TYPE_STORE): string
    {
        $identity = $this->scopeConfig->getValue('punchout/connection/sender_identity', $scope);

        return null === $identity ? '' : strtolower($identity);
    }

    public function getSharedSecret(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        $secret = $this->scopeConfig->getValue('punchout/connection/shared_secret', $scope);

        return $this->encryptor->decrypt($secret);
    }

    public function isExtendedLogging(string $scope = Data::SCOPE_TYPE_STORE): bool
    {
        return '1' == $this->scopeConfig->getValue('punchout/connection/extended_logging', $scope);
    }

    public function getUnitOfMeasure(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/item/unit_of_measure', $scope);
    }

    public function getClassificationDomain(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/item/classification_domain', $scope);
    }

    public function getClassificationCode(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/item/classification_code', $scope);
    }

    public function getProceedoUserName(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/order/username', $scope);
    }

    public function getProceedoPassword(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        $secret = $this->scopeConfig->getValue('punchout/order/password', $scope);

        return $this->encryptor->decrypt($secret);
    }

    public function getShippingMethod(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/order/shipping_method', $scope);
    }

    public function getPaymentMethod(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/order/payment_method', $scope);
    }

    public function getMaterialGroup(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/oci_item/material_group', $scope);
    }

    public function getUnit(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/oci_item/unit', $scope);
    }

    public function getLeadTime(string $scope = Data::SCOPE_TYPE_STORE): ?string
    {
        return $this->scopeConfig->getValue('punchout/oci_item/lead_time', $scope);
    }

    public function getCustomFieldValue(int $field, string $scope = Data::SCOPE_TYPE_STORE): string
    {
        $fieldValue = $this->scopeConfig->getValue('punchout/oci_item/custom'.$field, $scope);

        return null === $fieldValue ? '' : $fieldValue;
    }

    public function getStoreVatNumber(): string
    {
        $store = $this->storeManager->getStore();
        $info = $this->storeInformation->getStoreInformationObject($store);

        return $info->getVatNumber() ?? '';
    }

    public function getStoreName(): string
    {
        $store = $this->storeManager->getStore();
        $info = $this->storeInformation->getStoreInformationObject($store);

        return $info->getName() ?? '';
    }
}
