<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */

namespace Ecomero\PunchOut\Model\Csp;

use Magento\Csp\Api\PolicyCollectorInterface;
use Magento\Csp\Model\Policy\FetchPolicy;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

/**
 * Collects form-action CSP policies from admin configuration
 *
 * Allows PunchOut module to dynamically whitelist buyer system URLs
 * for form submissions without hardcoding them in csp_whitelist.xml
 *
 * @since 2.1.0
 */
class FormActionCollector implements PolicyCollectorInterface
{
    /** @var string Configuration path for form-action hosts */
    private const CONFIG_PATH_FORM_ACTION_HOSTS = 'punchout/security/csp_form_action_hosts';

    /**
     * @param ScopeConfigInterface $scopeConfig Scope configuration
     */
    public function __construct(
        private ScopeConfigInterface $scopeConfig
    ) {
    }

    /**
     * Collect form-action policies from configuration
     *
     * @param array<PolicyInterface> $defaultPolicies Existing policies
     * @return array<PolicyInterface> Merged policies including configured hosts
     * @since 2.1.0
     */
    public function collect(array $defaultPolicies = []): array
    {
        $policies = $defaultPolicies;
        $configuredHosts = $this->getConfiguredHosts();

        foreach ($configuredHosts as $host) {
            $policies[] = new FetchPolicy(
                'form-action',          // Policy ID
                false,                  // noneAllowed - 'none' not allowed
                [$host],                // hostSources - the buyer domain
                [],                     // schemeSources - no additional schemes
                false,                  // selfAllowed - 'self' not needed (external domain)
                false,                  // inlineAllowed - 'unsafe-inline' not allowed
                false,                  // evalAllowed - 'unsafe-eval' not allowed
                [],                     // nonceValues - no nonces
                [],                     // hashValues - no hashes
                false,                  // dynamicAllowed - dynamic not allowed
                false                   // eventHandlersAllowed - event handlers not allowed
            );
        }

        return $policies;
    }

    /**
     * Get configured hosts from admin configuration
     *
     * @return array<string> Array of hosts with https:// protocol
     * @since 2.1.0
     */
    private function getConfiguredHosts(): array
    {
        $hostsValue = $this->scopeConfig->getValue(
            self::CONFIG_PATH_FORM_ACTION_HOSTS,
            ScopeInterface::SCOPE_STORE
        );

        if (empty($hostsValue)) {
            return [];
        }

        // Parse comma-separated list
        $hosts = array_filter(array_map('trim', explode(',', $hostsValue)));
        
        // Add https:// protocol to each host for CSP
        return array_map(function($host) {
            // If wildcard, keep as is for CSP (e.g., *.example.com)
            if (strpos($host, '*') === 0) {
                return 'https://' . $host;
            }
            // For specific hosts, add https://
            return 'https://' . $host;
        }, $hosts);
    }
}
