<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Test\Integration\Model;

use Ecomero\PunchOut\Api\Data\SetupInterface;
use Ecomero\PunchOut\Api\SetupRepositoryInterface;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\TestFramework\Helper\Bootstrap;
use PHPUnit\Framework\TestCase;

/**
 * Integration test for SetupRepository
 *
 * Tests the full persistence layer including database operations
 *
 * @magentoAppArea frontend
 * @magentoDbIsolation enabled
 * @since 2.0.0
 */
class SetupRepositoryTest extends TestCase
{
    /**
     * @var SetupRepositoryInterface
     */
    private $repository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var FilterBuilder
     */
    private $filterBuilder;

    /**
     * Set up test dependencies
     *
     * @return void
     */
    protected function setUp(): void
    {
        parent::setUp();
        $objectManager = Bootstrap::getObjectManager();
        $this->repository = $objectManager->create(SetupRepositoryInterface::class);
        $this->searchCriteriaBuilder = $objectManager->create(SearchCriteriaBuilder::class);
        $this->filterBuilder = $objectManager->create(FilterBuilder::class);
    }

    /**
     * Test loading setup from valid cXML
     *
     * @return void
     */
    public function testLoadFromXml(): void
    {
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);

        $this->assertInstanceOf(SetupInterface::class, $setup);
        $this->assertEquals('RegionJonkoping', $setup->getHeaderSenderIdentity());
        $this->assertEquals('production', $setup->getRequestDeploymentMode());
        $this->assertEquals('create', $setup->getRequestOperation());
        $this->assertEquals('c208b7f4b1ebffcafb1c5c6563e44cc1', $setup->getRequestBuyerCookie());
    }

    /**
     * Test saving and retrieving setup
     *
     * @return void
     */
    public function testSaveAndGetById(): void
    {
        // Create a setup from XML
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);
        
        // Save it
        $savedSetup = $this->repository->save($setup);
        
        // Verify it has an ID
        $this->assertNotNull($savedSetup->getId());
        
        // Retrieve it by ID
        $retrievedSetup = $this->repository->getById($savedSetup->getId());
        
        // Verify data integrity
        $this->assertEquals($savedSetup->getToken(), $retrievedSetup->getToken());
        $this->assertEquals($savedSetup->getHeaderSenderIdentity(), $retrievedSetup->getHeaderSenderIdentity());
        $this->assertEquals($savedSetup->getRequestBuyerCookie(), $retrievedSetup->getRequestBuyerCookie());
    }

    /**
     * Test retrieving setup by token
     *
     * @return void
     */
    public function testGetByToken(): void
    {
        // Create and save a setup
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);
        $savedSetup = $this->repository->save($setup);
        
        $token = $savedSetup->getToken();
        
        // Retrieve by token
        $retrievedSetup = $this->repository->getByToken($token);
        
        $this->assertNotNull($retrievedSetup);
        $this->assertEquals($savedSetup->getId(), $retrievedSetup->getId());
        $this->assertEquals($token, $retrievedSetup->getToken());
    }

    /**
     * Test that non-existent token returns null
     *
     * @return void
     */
    public function testGetByNonExistentToken(): void
    {
        $result = $this->repository->getByToken('non-existent-token-12345');
        $this->assertNull($result);
    }

    /**
     * Test deleting setup
     *
     * @return void
     */
    public function testDelete(): void
    {
        // Create and save a setup
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);
        $savedSetup = $this->repository->save($setup);
        
        $setupId = $savedSetup->getId();
        
        // Delete it
        $this->repository->delete($savedSetup);
        
        // Verify it's gone
        $this->expectException(NoSuchEntityException::class);
        $this->repository->getById($setupId);
    }

    /**
     * Test deleting setup by ID
     *
     * @return void
     */
    public function testDeleteById(): void
    {
        // Create and save a setup
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);
        $savedSetup = $this->repository->save($setup);
        
        $setupId = $savedSetup->getId();
        
        // Delete by ID
        $this->repository->deleteById($setupId);
        
        // Verify it's gone
        $this->expectException(NoSuchEntityException::class);
        $this->repository->getById($setupId);
    }

    /**
     * Test getting list with search criteria
     *
     * @return void
     */
    public function testGetList(): void
    {
        // Create and save multiple setups
        for ($i = 0; $i < 3; $i++) {
            $xml = $this->getValidSetupXml();
            $setup = $this->repository->loadFromXml($xml);
            $this->repository->save($setup);
        }
        
        // Build search criteria
        $searchCriteria = $this->searchCriteriaBuilder
            ->setPageSize(10)
            ->setCurrentPage(1)
            ->create();
        
        // Get list
        $searchResults = $this->repository->getList($searchCriteria);
        
        $this->assertGreaterThanOrEqual(3, $searchResults->getTotalCount());
        $this->assertNotEmpty($searchResults->getItems());
    }

    /**
     * Test getting list with filters
     *
     * @return void
     */
    public function testGetListWithFilters(): void
    {
        // Create and save a setup with specific identity
        $xml = $this->getValidSetupXml();
        $setup = $this->repository->loadFromXml($xml);
        $setup->setHeaderSenderIdentity('TestIdentity123');
        $savedSetup = $this->repository->save($setup);
        
        // Build search criteria with filter
        $filter = $this->filterBuilder
            ->setField(SetupInterface::HEADER_SENDER_IDENTITY)
            ->setValue('TestIdentity123')
            ->setConditionType('eq')
            ->create();
        
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilters([$filter])
            ->create();
        
        // Get filtered list
        $searchResults = $this->repository->getList($searchCriteria);
        
        $this->assertGreaterThanOrEqual(1, $searchResults->getTotalCount());
        
        foreach ($searchResults->getItems() as $item) {
            $this->assertEquals('TestIdentity123', $item->getHeaderSenderIdentity());
        }
    }

    /**
     * Test that NoSuchEntityException is thrown for non-existent ID
     *
     * @return void
     */
    public function testGetByIdThrowsExceptionForNonExistent(): void
    {
        $this->expectException(NoSuchEntityException::class);
        $this->repository->getById('999999');
    }

    /**
     * Get valid cXML setup request for testing
     *
     * @return \SimpleXMLElement
     */
    private function getValidSetupXml(): \SimpleXMLElement
    {
        $xmlString = '<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE cXML SYSTEM "http://xml.cxml.org/schemas/cXML/1.2.041/cXML.dtd">
<cXML payloadID="1577107986.2072@punchoutcommerce.com" timestamp="2019-12-23T13:33:06+00:00">
  <Header>
    <From>
      <Credential domain="FromNetworkId">
        <Identity>FromId</Identity>
      </Credential>
    </From>
    <To>
      <Credential domain="DUNS">
        <Identity>ToId</Identity>
      </Credential>
    </To>
    <Sender>
      <Credential domain="SenderNetworkId">
        <Identity>RegionJonkoping</Identity>
        <SharedSecret>mySecret</SharedSecret>
      </Credential>
      <UserAgent>PunchOutCommerce PunchOut Tester</UserAgent>
    </Sender>
  </Header>
  <Request deploymentMode="production">
    <PunchOutSetupRequest operation="create">
      <BuyerCookie>c208b7f4b1ebffcafb1c5c6563e44cc1</BuyerCookie>
      <Extrinsic name="User">jdoe12345</Extrinsic>
      <BrowserFormPost>
        <URL>https://punchoutcommerce.com/tools/cxml-punchout-return</URL>
      </BrowserFormPost>
      <Contact>
        <Name xml:lang="en">John Doe</Name>
        <Email>jdoe@example.com</Email>
      </Contact>
      <SupplierSetup>
        <URL>http://punchout.example.com/punchout</URL>
      </SupplierSetup>
      <ShipTo>
        <Address addressID="TEST-ID">
          <Name xml:lang="en">TEST-Name</Name>
          <PostalAddress>
            <Street>123 Street Address</Street>
            <City>Rockville</City>
            <State>MD</State>
            <PostalCode>20850</PostalCode>
            <Country isoCountryCode="US">US</Country>
          </PostalAddress>
        </Address>
      </ShipTo>
    </PunchOutSetupRequest>
  </Request>
</cXML>';

        return new \SimpleXMLElement($xmlString);
    }

    /**
     * Clean up after tests
     *
     * @return void
     */
    protected function tearDown(): void
    {
        parent::tearDown();
        // Database isolation handles cleanup
    }
}
