# Content Security Policy (CSP) Configuration

## Overview

The Ecomero PunchOut module requires communication with external buyer systems to function properly. This involves:

- **Form Submissions**: Shopping cart data is POSTed to buyer's procurement systems
- **AJAX Requests**: Real-time data exchange with buyer APIs
- **Redirects**: User navigation back to buyer systems after shopping

By default, Magento 2.4.8's Content Security Policy (CSP) blocks all external connections for security. This module provides a flexible configuration system to whitelist trusted buyer domains **without hardcoding them**.

## Why CSP Matters

Content Security Policy is a security standard that prevents:
- Cross-site scripting (XSS) attacks
- Data injection attacks
- Unauthorized data exfiltration
- Malicious code execution

For PunchOut, you need to explicitly trust buyer systems to receive shopping cart data. This is done through **CSP whitelisting**.

## Configuration Location

Navigate to: **Stores > Configuration > eComero > PunchOut > Security (CSP)**

Two configuration fields are available:

### 1. Form Action Hosts (`form-action`)

Controls which domains can receive HTML form submissions.

**Use Case**: The shopping cart is submitted to buyer's system via POST when user clicks "Return to Procurement System"

**Example Hosts**:
```
buyer-portal.acme.com, procurement.example.net, *.buyersystem.org
```

### 2. Connect Source Hosts (`connect-src`)

Controls which domains can be called via AJAX/fetch/XMLHttpRequest.

**Use Case**: Real-time API calls to buyer systems for validation, status updates, or webhooks

**Example Hosts**:
```
api.buyer.com, webhook.procurement.net, *.buyersystem.org
```

## Host Format Rules

### ✅ Valid Formats

| Format | Example | Description |
|--------|---------|-------------|
| Single domain | `buyer.example.com` | Specific domain only |
| Subdomain wildcard | `*.example.com` | All subdomains of example.com |
| Domain with port | `buyer.local:8080` | Specific port (for dev/testing) |
| Multiple domains | `buyer1.com, buyer2.net` | Comma-separated list |
| Localhost | `localhost`, `localhost:3000` | Development only |

### ❌ Invalid Formats

| Format | Example | Why Invalid |
|--------|---------|-------------|
| With protocol | `https://buyer.com` | Protocol added automatically |
| With path | `buyer.com/api/v1` | CSP is host-level only |
| Mid-string wildcard | `buy*.example.com` | Wildcard must be at start |
| IP addresses | `192.168.1.1` | Use domain names |
| Spaces | `buyer .com` | No spaces allowed |

## Configuration Steps

### Step 1: Identify Buyer Systems

Collect URLs from PunchOut setup requests. These are typically found in:
- cXML: `<BrowserFormPost><URL>https://buyer.example.com/punchout/return</URL>`
- OCI: `hook_url` parameter
- Proceedo: Webhook endpoint URL

**Extract the hostname only**:
- ❌ `https://buyer.example.com/punchout/return`
- ✅ `buyer.example.com`

### Step 2: Configure in Admin

1. Go to **Stores > Configuration > eComero > PunchOut > Security (CSP)**
2. In **Form Action Hosts**, enter:
   ```
   buyer.example.com, procurement.partner.net
   ```
3. In **Connect Source Hosts**, enter (if using AJAX):
   ```
   api.buyer.example.com, webhook.partner.net
   ```
4. Click **Save Config**
5. **Flush cache**: `bin/magento cache:flush`

### Step 3: Test

1. Initiate a PunchOut session from buyer system
2. Add items to cart
3. Click "Return to Procurement System"
4. Verify cart data arrives at buyer system

**If form submission fails**, check browser console for CSP errors:
```
Refused to send form data to 'https://buyer.example.com/' because it violates the following Content Security Policy directive: "form-action 'self'".
```

**Solution**: Add `buyer.example.com` to Form Action Hosts configuration.

## Security Best Practices

### ✅ Do

- **Explicitly list known domains**: `buyer1.com, buyer2.net`
- **Use wildcards sparingly**: Only for subdomains you control
- **Document trusted domains**: Maintain list of approved buyers
- **Review regularly**: Remove domains for inactive buyers
- **Test in staging first**: Verify CSP before production
- **Monitor logs**: Check for CSP violations in browser console

### ❌ Don't

- **Use catch-all wildcards**: `*` or `*.com` defeats CSP purpose
- **Trust unknown domains**: Only add domains you've verified
- **Ignore CSP violations**: They indicate security issues or misconfigurations
- **Hardcode in csp_whitelist.xml**: Use admin config for flexibility
- **Include paths or protocols**: CSP operates at host level

## Wildcard Usage

### When to Use Wildcards

✅ **Good Use Case**: Buyer has multiple subdomains
```
*.buyer-system.com
```
Allows:
- `portal.buyer-system.com`
- `api.buyer-system.com`
- `staging.buyer-system.com`

❌ **Bad Use Case**: Too broad
```
*.com
```
This trusts **all** .com domains (millions of sites) - never do this!

### Wildcard Security

Wildcards inherit the parent domain's trust. Only use when:
1. You trust the parent organization
2. They control all subdomains
3. It's a known buyer/partner system

## Troubleshooting

### Problem: Form submission blocked by CSP

**Browser Console Error**:
```
Refused to send form data to 'https://buyer.example.com/return' because it violates 
the following Content Security Policy directive: "form-action 'self'".
```

**Solution**:
1. Extract hostname: `buyer.example.com`
2. Add to **Form Action Hosts** in admin
3. Clear cache: `bin/magento cache:flush`
4. Test again

### Problem: AJAX request blocked by CSP

**Browser Console Error**:
```
Refused to connect to 'https://api.buyer.com/webhook' because it violates 
the following Content Security Policy directive: "connect-src 'self'".
```

**Solution**:
1. Extract hostname: `api.buyer.com`
2. Add to **Connect Source Hosts** in admin
3. Clear cache
4. Test again

### Problem: Configuration not taking effect

**Checklist**:
- [ ] Cache cleared? (`bin/magento cache:flush`)
- [ ] Correct format? (no `https://`, no paths)
- [ ] Configuration saved? (check database: `core_config_data` table)
- [ ] Module enabled? (`bin/magento module:status Ecomero_PunchOut`)
- [ ] Browser cache cleared?

### Problem: Validation error when saving

**Error**: "Invalid CSP host(s): https://buyer.com"

**Cause**: Protocol included (https://)

**Solution**: Remove protocol, save only hostname: `buyer.com`

---

**Error**: "Invalid CSP host(s): buyer.com/api"

**Cause**: Path included (/api)

**Solution**: Remove path, save only hostname: `buyer.com`

## Technical Details

### How It Works

1. **Admin Configuration**: Store trusted hosts in `punchout/security/csp_form_action_hosts`
2. **Validation**: `CspHostList` backend model validates format before saving
3. **Collection**: `FormActionCollector` and `ConnectSrcCollector` read config
4. **Policy Generation**: Collectors create `FetchPolicy` objects with whitelisted hosts
5. **Injection**: Policies merged into Magento's CSP via `di.xml`
6. **Rendering**: CSP headers sent with every response

### CSP Header Example

After configuration, Magento sends:
```http
Content-Security-Policy: 
  form-action 'self' https://buyer.example.com https://procurement.partner.net;
  connect-src 'self' https://api.buyer.com https://webhook.partner.net;
```

### Files Involved

| File | Purpose |
|------|---------|
| `etc/adminhtml/system.xml` | Admin UI configuration fields |
| `Model/Config/Backend/CspHostList.php` | Input validation and normalization |
| `Model/Csp/FormActionCollector.php` | Collects form-action policies |
| `Model/Csp/ConnectSrcCollector.php` | Collects connect-src policies |
| `etc/di.xml` | Registers collectors with Magento CSP system |

## API for Programmatic Configuration

### Get Configured Hosts

```php
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

class MyClass
{
    public function __construct(
        private ScopeConfigInterface $scopeConfig
    ) {}

    public function getFormActionHosts(): array
    {
        $value = $this->scopeConfig->getValue(
            'punchout/security/csp_form_action_hosts',
            ScopeInterface::SCOPE_STORE
        );
        
        return $value ? array_map('trim', explode(',', $value)) : [];
    }
}
```

### Set via CLI (for automation)

```bash
# Set form-action hosts
bin/magento config:set punchout/security/csp_form_action_hosts \
  "buyer1.com,buyer2.net,*.buyer3.org"

# Set connect-src hosts  
bin/magento config:set punchout/security/csp_connect_src_hosts \
  "api.buyer1.com,webhook.buyer2.net"

# Clear cache
bin/magento cache:flush
```

## Multi-Store Configuration

CSP configuration is **global (default) scope only**. This is intentional because:
1. CSP is a security policy, should be consistent across stores
2. Buyer systems typically integrate at the Magento instance level
3. Simplifies management and reduces configuration errors

If you need store-specific CSP, consider:
- Using a reverse proxy (Varnish, Nginx) with store-specific headers
- Creating custom CSP collectors that read store-specific config

## Migration from Hardcoded csp_whitelist.xml

### Before (Hardcoded)

`etc/csp_whitelist.xml`:
```xml
<policies>
    <policy id="form-action">
        <values>
            <value id="buyer1" type="host">https://buyer1.com</value>
            <value id="buyer2" type="host">https://buyer2.net</value>
        </values>
    </policy>
</policies>
```

**Problems**:
- Requires code deployment to add/remove buyers
- Can't be changed by merchants
- Difficult to manage multiple buyers

### After (Dynamic Configuration)

**Admin Panel**: `Stores > Configuration > eComero > PunchOut > Security (CSP)`

**Form Action Hosts**:
```
buyer1.com, buyer2.net
```

**Benefits**:
- No code changes needed
- Merchants can self-manage
- Immediate updates (after cache flush)
- Easier to audit and review

## FAQ

**Q: Do I need to configure CSP for every PunchOut buyer?**

A: Yes, each buyer system domain must be whitelisted. This is a security requirement.

**Q: Can I use IP addresses instead of domain names?**

A: No, CSP requires domain names. If your buyer uses an IP, ask them to provide a domain name or set up DNS.

**Q: What if a buyer has multiple domains?**

A: Add all domains, comma-separated: `buyer-prod.com, buyer-staging.com, buyer-dev.local`

**Q: Is `*.com` allowed?**

A: Technically yes, but **never use it**. This would trust every .com domain on the internet, defeating CSP's purpose.

**Q: Do I need both form-action and connect-src?**

A: Depends on your implementation:
- **cXML/OCI/Proceedo**: Always need `form-action` (cart POSTed to buyer)
- **AJAX-based integrations**: Also need `connect-src` (JavaScript calls buyer APIs)

**Q: Will this break my site if misconfigured?**

A: CSP only blocks requests to non-whitelisted domains. Your Magento site continues working, but PunchOut returns may fail. Always test in staging first.

**Q: How do I find what domains need whitelisting?**

A: Check browser console (F12) for CSP violation errors. They show exact domains being blocked.

## Support & Resources

- **Magento CSP Documentation**: https://developer.adobe.com/commerce/php/development/security/content-security-policies/
- **CSP Specification**: https://www.w3.org/TR/CSP3/
- **CSP Validator**: https://csp-evaluator.withgoogle.com/
- **Module Documentation**: See README.md

---

**Last Updated**: October 15, 2025  
**Module Version**: 2.1.0+  
**Magento Compatibility**: 2.4.8+
