<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Helper;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Driver\File;
use Psr\Log\LoggerInterface;

/**
 * Custom logger for PunchOut module
 * 
 * Logs to custom file: var/log/eComero/punchout-YYYYMMDD.log
 * Automatically cleans up old log files based on retention policy
 * 
 * @since 2.0.0
 */
class Logger
{
    protected string $logRoot;

    public const FILE_PREFIX = 'punchout-';

    public const RETENTION_1_DAY = 60 * 60 * 24 * 1;
    public const RETENTION_7_DAY = 60 * 60 * 24 * 7;
    public const RETENTION_14_DAY = 60 * 60 * 24 * 14;
    public const RETENTION_30_DAY = 60 * 60 * 24 * 30;
    public const RETENTION_90_DAY = 60 * 60 * 24 * 90;

    /**
     * @param LoggerInterface $logger PSR-3 logger
     * @param Filesystem $filesystem Magento filesystem
     * @param File $fileSystem File driver
     * @param Data $data Configuration helper
     */
    public function __construct(
        protected LoggerInterface $logger,
        protected Filesystem $filesystem,
        protected File $fileSystem,
        protected Data $data
    ) {
        // Initialize log directory
        $varDir = $this->filesystem->getDirectoryRead(DirectoryList::VAR_DIR);
        $this->logRoot = $varDir->getAbsolutePath('log/eComero');
        
        // Create log directory if it doesn't exist
        if (!$this->fileSystem->isDirectory($this->logRoot)) {
            try {
                $this->fileSystem->createDirectory($this->logRoot, 0775);
            } catch (\Exception $e) {
                // Fallback to standard Magento log if custom directory can't be created
                $this->logRoot = $varDir->getAbsolutePath('log');
            }
        }
    }

    /**
     * Delete old log files based on retention policy
     * 
     * @return void
     * @since 2.0.0
     */
    public function deleteOldLogfiles(): void
    {
        $timeLimit = self::RETENTION_30_DAY;

        try {
            if (!$this->fileSystem->isDirectory($this->logRoot)) {
                return;
            }

            foreach ($this->fileSystem->readDirectory($this->logRoot) as $fileName) {
                if (strpos($fileName, $this->logRoot . '/' . self::FILE_PREFIX) === 0) {
                    $fileInfo = $this->fileSystem->stat($fileName);
                    $mtime = $fileInfo['mtime'];
                    $now = time();
                    
                    if ($now - $mtime >= $timeLimit && $timeLimit !== 0) {
                        $this->fileSystem->deleteFile($fileName);
                    }
                }
            }
        } catch (\Exception $e) {
            // Silently fail if cleanup fails
            $this->logger->warning('Failed to clean up old PunchOut log files: ' . $e->getMessage());
        }
    }

    /**
     * Get real IP address from request
     * 
     * @return string Client IP address
     * @since 2.0.0
     */
    private function getRealIpAddr(): string
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        }
        
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }

    /**
     * Get active log file path
     * 
     * @return string Full path to today's log file
     * @since 2.0.0
     */
    public function getActiveLogFile(): string
    {
        return $this->logRoot . '/' . self::FILE_PREFIX . date("Ymd") . '.log';
    }

    /**
     * Log error message
     * 
     * @param string $message Error message to log
     * @return void
     * @since 2.0.0
     */
    public function error(string $message): void
    {
        $context = ['ip' => $this->getRealIpAddr()];
        $this->logger->error('[PunchOut] ' . $message, $context);
        
        // Also write to custom log file
        $this->writeToFile('ERROR: ' . $message);
    }

    /**
     * Log info message (only if extended logging is enabled)
     * 
     * @param string $message Info message to log
     * @return void
     * @since 2.0.0
     */
    public function info(string $message): void
    {
        if ($this->data->isExtendedLogging(Data::SCOPE_TYPE_STORE)) {
            $context = ['ip' => $this->getRealIpAddr()];
            $this->logger->info('[PunchOut] ' . $message, $context);
            
            // Also write to custom log file
            $this->writeToFile('INFO: ' . $message);
        }
    }

    /**
     * Write message to custom log file
     * 
     * @param string $message Message to write
     * @return void
     * @since 2.0.0
     */
    private function writeToFile(string $message): void
    {
        try {
            $logFile = $this->getActiveLogFile();
            $timestamp = date('Y-m-d H:i:s');
            $ip = $this->getRealIpAddr();
            $logEntry = "[{$timestamp}] [{$ip}] {$message}" . PHP_EOL;
            
            // Append to file
            $handle = fopen($logFile, 'a');
            if ($handle) {
                fwrite($handle, $logEntry);
                fclose($handle);
            }
        } catch (\Exception $e) {
            // Silently fail if file write fails - PSR logger still has the message
        }
    }
}
