<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */

namespace Ecomero\PunchOut\Model\Config\Backend;

use Magento\Framework\App\Config\Value;
use Magento\Framework\Exception\ValidatorException;

/**
 * Backend model for CSP host list configuration
 *
 * Validates and normalizes comma-separated list of domains for CSP policies
 *
 * @since 2.1.0
 */
class CspHostList extends Value
{
    /**
     * Validate and normalize the CSP host list before saving
     *
     * @return $this
     * @throws ValidatorException If host list contains invalid domains
     * @since 2.1.0
     */
    public function beforeSave(): self
    {
        $value = $this->getValue();
        
        if (empty($value)) {
            // Empty list is valid - means no CSP exceptions configured
            return parent::beforeSave();
        }

        // Parse comma-separated list
        $hosts = array_map('trim', explode(',', $value));
        $validHosts = [];
        $invalidHosts = [];

        foreach ($hosts as $host) {
            if (empty($host)) {
                continue; // Skip empty entries
            }

            // Validate host format
            if ($this->isValidHost($host)) {
                $validHosts[] = $this->normalizeHost($host);
            } else {
                $invalidHosts[] = $host;
            }
        }

        if (!empty($invalidHosts)) {
            throw new ValidatorException(
                __(
                    'Invalid CSP host(s): %1. Hosts must be valid domain names (e.g., example.com, *.example.com). '
                    . 'Do not include protocols (http://, https://) or paths (/path).',
                    implode(', ', $invalidHosts)
                )
            );
        }

        // Store normalized, comma-separated list
        $this->setValue(implode(',', $validHosts));

        return parent::beforeSave();
    }

    /**
     * Validate if host is a valid domain name or wildcard pattern
     *
     * Accepts:
     * - example.com
     * - subdomain.example.com
     * - *.example.com (wildcard subdomain)
     * - localhost (for development)
     *
     * Rejects:
     * - http://example.com (has protocol)
     * - example.com/path (has path)
     * - example.com:8080 (has port - ports should be included with host)
     * - Invalid characters
     *
     * @param string $host Host to validate
     * @return bool True if valid
     * @since 2.1.0
     */
    private function isValidHost(string $host): bool
    {
        // Check for protocol (should not be included)
        if (preg_match('#^https?://#i', $host)) {
            return false;
        }

        // Check for path (should not be included)
        if (strpos($host, '/') !== false) {
            return false;
        }

        // Allow wildcards in subdomain position only
        if (strpos($host, '*') !== false) {
            // Wildcard must be at the start and followed by a dot
            if (!preg_match('/^\*\./', $host)) {
                return false;
            }
            // Remove wildcard for further validation
            $host = substr($host, 2);
        }

        // Allow localhost for development
        if ($host === 'localhost') {
            return true;
        }

        // Allow localhost with port
        if (preg_match('/^localhost:\d+$/', $host)) {
            return true;
        }

        // Validate domain format (with optional port)
        // Domain: alphanumeric with hyphens, dots for subdomains, optional :port
        $pattern = '/^([a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?\.)*[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?(:\d{1,5})?$/i';
        
        return (bool) preg_match($pattern, $host);
    }

    /**
     * Normalize host to consistent format
     *
     * - Converts to lowercase
     * - Trims whitespace
     * - Removes trailing dots
     *
     * @param string $host Host to normalize
     * @return string Normalized host
     * @since 2.1.0
     */
    private function normalizeHost(string $host): string
    {
        $host = strtolower(trim($host));
        $host = rtrim($host, '.');
        
        return $host;
    }

    /**
     * Parse stored value into array of hosts
     *
     * @return array<string> Array of validated hosts
     * @since 2.1.0
     */
    public function getHostsArray(): array
    {
        $value = $this->getValue();
        
        if (empty($value)) {
            return [];
        }

        return array_filter(array_map('trim', explode(',', $value)));
    }
}
