<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\PunchOut\Test\Unit\Model;

use Ecomero\PunchOut\Model\ResourceModel\Setup\CollectionFactory;
use Ecomero\PunchOut\Model\Setup;
use Ecomero\PunchOut\Model\SetupFactory;
use Ecomero\PunchOut\Model\SetupItemFactory;
use Ecomero\PunchOut\Model\SetupItemRepository;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchResultsInterfaceFactory;

class SetupRepositoryTest extends \PHPUnit\Framework\TestCase
{
    protected $setupRepositoryClass;
    protected $objectManager;

    const XML_DATA = '<?xml version="1.0" encoding="UTF-8"?>
    <!DOCTYPE cXML SYSTEM "http://xml.cxml.org/schemas/cXML/1.2.041/cXML.dtd">
    <cXML payloadID="1577107986.2072@punchoutcommerce.com" timestamp="2019-12-23T13:33:06+00:00">
      <Header>
        <From>
          <Credential domain="FromNetworkId">
            <Identity>FromId</Identity>
          </Credential>
        </From>
        <To>
          <Credential domain="DUNS">
            <Identity>ToId</Identity>
          </Credential>
        </To>
        <Sender>
          <Credential domain="SenderNetworkId">
            <Identity>RegionJonkoping</Identity>
            <SharedSecret>mySecret</SharedSecret>
          </Credential>
          <UserAgent>PunchOutCommerce PunchOut Tester</UserAgent>
        </Sender>
      </Header>
      <Request deploymentMode="production">
        <PunchOutSetupRequest operation="create">
          <BuyerCookie>c208b7f4b1ebffcafb1c5c6563e44cc1</BuyerCookie>
          <Extrinsic name="User">jdoe12345</Extrinsic>
          <Extrinsic name="UniqueUsername">jdoe12345</Extrinsic>
          <Extrinsic name="UserId">12345</Extrinsic>
          <Extrinsic name="UserEmail">jdoe@example.com</Extrinsic>
          <Extrinsic name="UserFullName">John Doe</Extrinsic>
          <Extrinsic name="UserPrintableName">John Doe</Extrinsic>
          <Extrinsic name="FirstName">John</Extrinsic>
          <Extrinsic name="LastName">Doe</Extrinsic>
          <Extrinsic name="PhoneNumber">555-555-5555</Extrinsic>
          <BrowserFormPost>
            <URL>https://punchoutcommerce.com/tools/cxml-punchout-return</URL>
          </BrowserFormPost>
          <SupplierSetup>
            <URL>http://punchout.snojken.com/index.php/rest/default/V1/punchout</URL>
          </SupplierSetup>
          <ShipTo>
            <Address addressID="TEST-ID">
              <Name xml:lang="en">TEST-Name</Name>
              <PostalAddress>
                <Street>123 Street Address</Street>
                <City>Rockville</City>
                <State>MD</State>
                <PostalCode>20850</PostalCode>
                <Country isoCountryCode="US">US</Country>
              </PostalAddress>
            </Address>
          </ShipTo>
          <ItemOut quantity="2">
            <ItemID>
              <SupplierPartID>RJB_Tsh1-DM</SupplierPartID>
              <SupplierPartAuxiliaryID>357</SupplierPartAuxiliaryID>
            </ItemID>
          </ItemOut>
          <ItemOut quantity="3">
            <ItemID>
              <SupplierPartID>RJB_Keps1</SupplierPartID>
              <SupplierPartAuxiliaryID>296</SupplierPartAuxiliaryID>
            </ItemID>
          </ItemOut>
        </PunchOutSetupRequest>
      </Request>
    </cXML>
    ';

    public function setUp() : void
    {
        $this->objectManager = new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);

        /** @var SetupFactory&\PHPUnit\Framework\MockObject\MockObject $setupFactoryMock */
        $setupFactoryMock = $this->getMockBuilder(SetupFactory::class)
            ->onlyMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();

        $setupFactoryMock->method('create')->willReturn(
            $this->objectManager->getObject('Ecomero\PunchOut\Model\Setup')
        );

        /** @var SetupItemFactory&\PHPUnit\Framework\MockObject\MockObject $setupItemFactoryMock */
        $setupItemFactoryMock = $this->getMockBuilder(SetupItemFactory::class)
          ->disableOriginalConstructor()
          ->getMock();

        /** @var CollectionFactory&\PHPUnit\Framework\MockObject\MockObject $collectionFactoryMock */
        $collectionFactoryMock = $this->getMockBuilder(CollectionFactory::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var SearchResultsInterfaceFactory&\PHPUnit\Framework\MockObject\MockObject $searchResultsFactoryMock */
        $searchResultsFactoryMock = $this->getMockBuilder(SearchResultsInterfaceFactory::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var CollectionProcessorInterface&\PHPUnit\Framework\MockObject\MockObject $collectionProcessorMock */
        $collectionProcessorMock = $this->getMockBuilder(CollectionProcessorInterface::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var SetupItemRepository&\PHPUnit\Framework\MockObject\MockObject $setupItemRepositoryMock */
        $setupItemRepositoryMock = $this->getMockBuilder(SetupItemRepository::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var SearchCriteriaBuilder&\PHPUnit\Framework\MockObject\MockObject $searchCriteriaBuilder */
        $searchCriteriaBuilder = $this->getMockBuilder(SearchCriteriaBuilder::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->setupRepositoryClass = new \Ecomero\PunchOut\Model\SetupRepository(
            $setupFactoryMock,
            $setupItemFactoryMock,
            $collectionFactoryMock,
            $searchResultsFactoryMock,
            $collectionProcessorMock,
            $setupItemRepositoryMock,
            $searchCriteriaBuilder
        );
    }

    public function testLoadFromXml() : void
    {
        $xml = new \SimpleXMLElement(SetupRepositoryTest::XML_DATA);

        /** @var \Ecomero\PunchOut\Model\Setup $rc */
        $rc =  $this->setupRepositoryClass->loadFromXml($xml);
        $this->assertEquals('1577107986.2072@punchoutcommerce.com', $rc->getPayloadId());
        $this->assertEquals(new \DateTime('2019-12-23T13:33:06+00:00'), $rc->getTimestamp());
        $this->assertEquals('FromNetworkId', $rc->getHeaderFromDomain());
        $this->assertEquals('FromId', $rc->getHeaderFromIdentity());
        $this->assertEquals('DUNS', $rc->getHeaderToDomain());
        $this->assertEquals('ToId', $rc->getHeaderToIdentity());
        $this->assertEquals('SenderNetworkId', $rc->getHeaderSenderDomain());
        $this->assertEquals('RegionJonkoping', $rc->getHeaderSenderIdentity());
        $this->assertEquals('PunchOutCommerce PunchOut Tester', $rc->getHeaderSenderUserAgent());
        $this->assertEquals('production', $rc->getRequestDeploymentMode());
        $this->assertEquals('create', $rc->getRequestOperation());
        $this->assertEquals('c208b7f4b1ebffcafb1c5c6563e44cc1', $rc->getRequestBuyerCookie());
        $this->assertEquals(
            'https://punchoutcommerce.com/tools/cxml-punchout-return',
            $rc->getRequestBrowserFormPostUrl()
        );
        $this->assertEquals(
            'http://punchout.snojken.com/index.php/rest/default/V1/punchout',
            $rc->getRequestSupplierSetupUrl()
        );

        $a = $rc->getRequestExtrinsic();
        $extrinsic = unserialize($a);
        $this->assertEquals($extrinsic['User'], 'jdoe12345');
        $this->assertEquals($extrinsic['UniqueUsername'], 'jdoe12345');
        $this->assertEquals($extrinsic['UserId'], '12345');
        $this->assertEquals($extrinsic['UserEmail'], 'jdoe@example.com');
        $this->assertEquals($extrinsic['UserFullName'], 'John Doe');

        $this->assertEquals('TEST-ID', $rc->getRequestShipToId());
        $this->assertEquals('TEST-Name', $rc->getRequestShipToName());
        $this->assertEquals('123 Street Address', $rc->getRequestShipToStreet());
        $this->assertEquals('Rockville', $rc->getRequestShipToCity());
        $this->assertEquals('MD', $rc->getRequestShipToState());
        $this->assertEquals('20850', $rc->getRequestShipToPostalCode());
        $this->assertEquals('US', $rc->getRequestShipToCountry());
    }
}
