<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\Pyramid\Model\Erp;

use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Ecomero\ErpCore\Model\Order\Response as OrderResponse;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Ecomero\Pyramid\Model\PyramidAdapter;
use Ecomero\Pyramid\Model\PyramidAPI;
use InvalidArgumentException;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class PyramidOrder extends PyramidAdapter implements ErpOrderInterface
{
    public function __construct(
        PyramidAPI $pyramidAPI,
        Settings $settings,
        Capability $capability,
        ArrayManager $arrayManager,
        StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory
    ) {
        parent::__construct(
            $pyramidAPI,
            $settings,
            $capability,
            $arrayManager,
            $storeManager,
            $orderResponseFactory
        );
    }

    public function getProfitCenters(): array
    {
        return [];
    }

    public function postOrder(string $salesOrderId): void
    {
    }

    public function createDocument(
        string $docType,
        ?string $customerId,
        ?string $companyId,
        string $currency,
        \Magento\Sales\Model\Order\Address $billingAddress,
        \Magento\Sales\Model\Order\Address $shippingAddress,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        string $magentoOrderId,
        ?string $extPONumber,
        string $salesPerson,
        ?string $profitCenter,
        ?string $termsOfDelivery,
        ?string $wayOfDelivery,
        ?string $customerCategory,
        ?string $customerDistrict
    ): OrderResponse {
        $this->currentOrder['salesperson'] = $salesPerson;
        $this->currentOrder['profitCenter'] = $profitCenter;
        $this->currentOrder['termsOfDelivery'] = $termsOfDelivery;
        $this->currentOrder['wayOfDelivery'] = $wayOfDelivery;
        $this->currentOrder['customerCategory'] = $customerCategory;
        $this->currentOrder['customerDistrict'] = $customerDistrict;

        $this->currentOrder['customerNumber'] = $customerId;
        $this->currentOrder['companyNumber'] = $companyId;
        $this->currentOrder['currencyCode'] = $currency;
        $this->currentOrder['externalDocument'] = $magentoOrderId;
        $this->currentOrder['customerPONumber'] = $extPONumber;
        $this->currentOrder['vatNumber'] = $billingAddress->getVatId();

        // Billing address
        $this->currentOrder['name'] = $billingAddress->getCompany();
        $this->currentOrder['reference'] = $billingAddress->getFirstname().' '.$billingAddress->getLastname();
        $this->currentOrder['telephone'] = $billingAddress->getTelephone();
        $this->currentOrder['street'] = '';
        $this->currentOrder['street2'] = '';

        if (count($billingAddress->getStreet()) >= 1) {
            $this->currentOrder['street'] = $billingAddress->getStreetLine(1);
        }
        if (count($billingAddress->getStreet()) >= 2) {
            $this->currentOrder['street2'] = $billingAddress->getStreetLine(2);
        }
        $this->currentOrder['city'] = $billingAddress->getCity();
        $this->currentOrder['countryLetterCode'] = $billingAddress->getCountryId();
        $this->currentOrder['postalCode'] = $billingAddress->getPostcode();
        $this->currentOrder['GLN'] = '';

        // Delivery address
        $this->currentOrder['deliveryName'] = $shippingAddress->getCompany();
        $this->currentOrder['deliveryReference'] = $shippingAddress->getFirstname().' '.$shippingAddress->getLastname();
        $this->currentOrder['deliveryTelephone'] = $shippingAddress->getTelephone();
        $this->currentOrder['deliveryStreet'] = '';
        $this->currentOrder['deliveryStreet2'] = '';

        if (count($shippingAddress->getStreet()) >= 1) {
            $this->currentOrder['deliveryStreet'] = $shippingAddress->getStreetLine(1);
        }
        if (count($shippingAddress->getStreet()) >= 2) {
            $this->currentOrder['deliveryStreet2'] = $shippingAddress->getStreetLine(2);
        }
        $this->currentOrder['deliveryCity'] = $shippingAddress->getCity();
        $this->currentOrder['deliveryCountryLetterCode'] = $shippingAddress->getCountryId();
        $this->currentOrder['deliveryPostalCode'] = $shippingAddress->getPostcode();
        $this->currentOrder['deliveryGLN'] = '';

        $this->currentOrder['orderLines'] = [];
        $this->currentOrder['number'] = '';
        $this->currentOrder['id'] = '';

        return $this->orderResponseFactory->create([
            'response' => $this->currentOrder,
        ]);
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent,
        ?string $comment,
        ?string $locationId
    ): void {
        $discountIsAmount = false;
        $discount = 0;
        if ($discountAmount > 0) {
            $discountIsAmount = true;
            $discount = $discountAmount;
        } else {
            $discountIsAmount = false;
            $discount = $discountPercent;
        }

        if (null !== $comment) {
            $finalComment = '';
            $commentLines = explode("\n", $comment);
            foreach ($commentLines as $line) {
                while (mb_strlen($line) > self::MAX_COMMENT_LEN) {
                    $finalComment .= mb_substr($line, 0, self::MAX_COMMENT_LEN)."\n";
                    $line = mb_substr($line, self::MAX_COMMENT_LEN, mb_strlen($line));
                }
                $finalComment .= $line."\n";
            }
            $comment = trim($finalComment);
        }

        $this->currentOrder['orderLines'][] = [
            'articleNumber' => $erpItemId,
            'qty' => $qtyOrdered,
            'price' => $price,
            'discount' => $discount,
            'discountIsAmount' => $discountIsAmount,
            'amount' => $price * $qtyOrdered,
            'vatCode' => $vatCode,
            'comment' => $comment ?? '',
        ];
    }

    public function updateDocument(
        string $docType,
        string $erpId,
        string $magentoOrderId
    ): OrderResponse {
        $this->currentOrder['externalDocument'] = $magentoOrderId;

        $response = $this->pyramidAPI->post('command=createorder', $this->currentOrder);

        return $this->orderResponseFactory->create([
            'response' => $response,
        ]);
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ): object {
        $tmpResult = [];

        if (0 === strpos($magentoOrderId, 'PENDING(')) {
            $magentoOrderId = str_replace('PENDING(', '', $magentoOrderId);
            $magentoOrderId = rtrim($magentoOrderId, ')');
        }

        $command = strtolower($docType);
        $rc = $this->pyramidAPI->get("command={$command}&externalDocumentNumber={$magentoOrderId}");
        if (1 == count($rc)) {
            $tmpResult = $rc[0];

            if (!empty($tmpResult['deliveryDate'])) {
                $date = $this->convertToDate($tmpResult['deliveryDate']);
                $tmpResult['deliveryDate'] = $date;
            }

            return (object) $tmpResult;
        }

        return (object) [];
    }

    public function getSalesPersons(): array
    {
        return [[
            'code' => '1',
            'name' => 'Adobe Commerce',
        ]];
    }

    private function convertToDate(string $deliveryDate): \DateTime
    {
        $date = new \DateTime();

        $deliveryYear = substr($deliveryDate, 0, 4);

        if (!$deliveryYear) {
            throw new InvalidArgumentException('Could not parse year from delivery date');
        }

        $deliveryWeek = substr($deliveryDate, 4);
        $deliveryDay = '';

        if (strlen($deliveryWeek) > 2) {
            $deliveryDay = substr($deliveryWeek, 2);
            $deliveryWeek = substr($deliveryWeek, 0, 2);
        }

        if (!$deliveryWeek) {
            throw new InvalidArgumentException('Could not parse year from delivery date');
        }

        $date->setISODate((int) $deliveryYear, (int) $deliveryWeek, (int) $deliveryDay ?: 1);

        return $date;
    }
}
