<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\Pyramid\Model\Erp;

use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Ecomero\ErpCore\Model\ResourceModel\Item\Collection;
use Ecomero\Pyramid\Helper\Data as Settings;
use Ecomero\Pyramid\Model\PyramidAdapter;
use Ecomero\Pyramid\Model\PyramidAPI;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class PyramidCatalog extends PyramidAdapter implements ErpCatalogInterface
{
    protected $resourceConnection;
    protected $itemListCache = [];

    protected const SQL_CLEAR_COST = <<<SQL
        DELETE	d
        FROM 	catalog_product_entity_decimal d 
                JOIN eav_attribute a ON a.attribute_id = d.attribute_id
        WHERE 	attribute_code = 'cost' AND d.store_id <> 0
SQL;

    protected const SQL_GET_EUR_STORES = <<<SQL
        SELECT  store_id
        FROM    store
        WHERE   code like '%eur%'
SQL;

    public function __construct(
        PyramidAPI $pyramidAPI,
        Settings $settings,
        Capability $capability,
        ArrayManager $arrayManager,
        StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory,
        ResourceConnection $resourceConnection
    ) {
        parent::__construct(
            $pyramidAPI,
            $settings,
            $capability,
            $arrayManager,
            $storeManager,
            $orderResponseFactory
        );
        $this->resourceConnection = $resourceConnection;
    }

    public function setItemImportCompleted(): void
    {
    }

    public function getItemPrices(string $category): array
    {
        $result = [];

        $priceListCode = $this->storeManager->getWebsite($this->getWebsite())->getCode();
        $response = $this->pyramidAPI->get("command=pricelistcurrency&category={$category}");
        foreach ($response as $item) {
            if (strtolower($this->settings->getLCY()) === strtolower($priceListCode)) {
                $item['website'] = $this->getWebsite();
                $item['priceCurrency'] = $item['lcyPrice'];
                $result[] = $item;
            } elseif (strtolower($item['currency']) === strtolower($priceListCode)) {
                $item['website'] = $this->getWebsite();
                $result[] = $item;
            }
        }

        return $result;
    }

    public function getItemPriceLists(): array
    {
        $rc = [];
        $response = $this->pyramidAPI->get('command=pricelist');
        $combinations = $this->arrayManager->get('priceAgreements', $response);
        foreach ($combinations as $combination) {
            $pId = $combination['price_list_id'];
            $aId = $combination['agreement_id'];
            $priceList = $this->getPriceList($response, $pId);
            $agreementList = $this->getPriceList($response, $aId);

            if ($priceList || $agreementList) {
                // Found valid price list

                $pName = $this->arrayManager->get('name', $priceList) ?? '';
                $aName = $this->arrayManager->get('name', $agreementList) ?? '';

                $name = $this->getPriceListName($pName, $aName);
                $pPrices = $this->arrayManager->get('prices', $priceList) ?? [];
                $aPrices = $this->arrayManager->get('prices', $agreementList) ?? [];
                $paPrices = array_merge($pPrices, $aPrices);

                $combinedPrices = array_filter($paPrices, function ($price) {
                    return $price['sku'] || (
                        $price['article_type']
                                && self::DISCOUNT_BASE_PRICE == $price['discount_base']
                                && self::PRICE_TYPE_DISCOUNT == $price['price_type']
                    );
                });

                $priceList = [
                    'id' => $pId . '_' . $aId,
                    'name' => $name,
                    'prices' => $combinedPrices,
                ];

                $rc[] = $priceList;
            }
        }

        return $rc;
    }

    public function getItemAttributes(string $category): array
    {
        $result = [];
        foreach ($this->itemListCache as $item) {
            $result[] = $this->mapAttribute('category', 'article_type', $item, 'Option');
            $result[] = $this->mapAttribute('leadTime', 'lead_time', $item);
            $result[] = $this->mapAttribute('producibleQty', 'qty_producible', $item);
            $result[] = $this->mapAttribute('producibleQtyUpdated', 'qty_producible_updated', $item);
            $result[] = $this->mapAttribute('hsCode', 'hs_code', $item);
            $result[] = $this->mapAttribute('height', 'height', $item);
            $result[] = $this->mapAttribute('length', 'length', $item);
            $result[] = $this->mapAttribute('width', 'width', $item);
            $result[] = $this->mapAttribute('grossWeight', 'gross_weight', $item);
            $result[] = $this->mapAttribute('packageSize', 'package_size', $item);
            $result[] = $this->mapAttribute('coutryOfManufacture', 'country_of_manufacture', $item, 'Option');

            $formattedDescriptions = $this->arrayManager->get('formattedDescription', $item);
            foreach ($formattedDescriptions as $formattedDescription) {
                $attribute = [];
                $attribute['sku'] = $item['sku'];
                $attribute['locale'] = $formattedDescription['language'];
                $attribute['name'] = 'description';
                $attribute['type'] = 'Text';
                $attribute['value'] = strip_tags($formattedDescription['description'], ['br', 'p']);
                $result[] = $attribute;
            }
        }

        return array_filter($result);
    }

    public function getNonInventoryItems(): array
    {
        return [];
    }

    public function addPicture(string $itemId, string $imageUrl): int
    {
        return 0;
    }

    public function createCommonName(Collection $itemCollection): bool
    {
        return false;
    }

    public function getItemList(
        string $category,
        bool $force
    ): array {
        $pageSize = 0; // Disable paging, not supported on Zen V12 (only from V14)
        $row = 0;

        do {
            $endRow = $row + $pageSize;
            $queryParam = "command=article&filter={$category}&pageRow={$row}&pageSize={$pageSize}&includeDescriptions=true";
            $pageData = $this->pyramidAPI->get($queryParam);
            $this->itemListCache = array_merge($this->itemListCache, $pageData);
            $row += count($pageData);
        } while (count($pageData) === $pageSize && 0 !== $pageSize);

        foreach ($this->itemListCache as &$item) {
            $item['commonName'] = $this->getCommonName($item);
        }

        return $this->itemListCache;
    }

    public function getErpExtVersion(): string
    {
        $response = $this->pyramidAPI->get('command=version');

        return $this->arrayManager->get('version', $response);
    }

    private function mapAttribute(string $name, string $attributeCode, array $item, string $type = 'Text'): ?array
    {
        $attribute = null;

        $sku = (string) $this->arrayManager->get('sku', $item);
        $value = (string) $this->arrayManager->get($name, $item);

        if ($sku && null !== $value && '' !== $value && 0 !== $value && '0' !== $value) {
            $attribute = [];
            $attribute['sku'] = $item['sku'];
            $attribute['name'] = $attributeCode;
            $attribute['type'] = $type;
            $attribute['value'] = $item[$name];
        }

        return $attribute;
    }

    private function getPriceListName(string $pName, string $aName): string
    {
        $name = '';

        if ($pName && $aName) {
            $name = "{$pName}, {$aName}";
        } elseif ($pName) {
            $name = $pName;
        } else {
            $name = $aName;
        }

        return $name;
    }

    private function getPriceList(array $response, string $priceListId): array
    {
        $priceLists = $this->arrayManager->get('priceLists', $response);
        foreach ($priceLists as $priceList) {
            if ($priceList['id'] === $priceListId) {
                return $priceList;
            }
        }

        return [];
    }

    private function getCommonName(array $item): ?string
    {
        return $this->arrayManager->get('sku', $item);
    }

    public function postProcessing(string $errorMessage): string
    {
        // Recalculate cost after product import for EUR sites
        $exchangeRate = $this->settings->getExchangeRate();
        $adapter = $this->resourceConnection->getConnection();

        $adapter->query(self::SQL_CLEAR_COST);
        $allEurStores = $adapter->fetchCol(self::SQL_GET_EUR_STORES);

        foreach ($allEurStores as $store_id) {
            $sql = <<<SQL
                INSERT INTO catalog_product_entity_decimal(attribute_id, row_id, store_id, value)
                    SELECT	a.attribute_id, d.row_id, {$store_id}, d.value / ?
                    FROM 	catalog_product_entity_decimal d 
                            JOIN eav_attribute a ON a.attribute_id = d.attribute_id
                    WHERE 	attribute_code = 'cost' AND d.store_id = 0
SQL;
            $adapter->query($sql, [$exchangeRate]);
        }
        return $errorMessage;
    }
}
